/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2016 Univ. Grenoble Alpes, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/


// PML
#include <pml/PhysicalModel.h>
#include <pml/MultiComponent.h>

// LML
#include <lml/Loads.h>
#include <lml/Load.h>

// Physical Model Component
#include "LoadsManager.h"
#include "LoadsSimulation.h"
#include "LoadsMovie.h"
#include "LoadsEditor.h"
#include "AtomDecoration.h"
#include "PMManagerDC.h"
#include "AtomDC.h"

// camiTK
#include <Application.h>
#include <MainWindow.h>
#include <InteractiveViewer.h>
using namespace camitk;

// Qt
#include <QDockWidget>
#include <QMessageBox>
#include <QFileDialog>

// other
#include <math.h>
#include <limits>

//----------------- constructor ----------------------------
LoadsManager::LoadsManager(PMManagerDC *pmManagerDC) {
    fileName = "";
    motorAddonLocation = "";
    myPMManagerDC = pmManagerDC;
    myLoads = NULL;
    previewDialog = NULL;
    loadsEditor = NULL;
    changed = false;
    loadsChangedFlag = false;
    showLoads = false;
    atomDataDisplay = NONE;
    referencePM = NULL;
    constrainedAtomDataScale = false;
    min = std::numeric_limits<double>::max();
    max = -1.0;

// automatically create the dialog, but do not show yet (that will create the new toolbar)
    simulationDialog = new LoadsSimulation(this, Application::getMainWindow());
    simDock = new QDockWidget(Application::getMainWindow());
    simDock->setAllowedAreas(Qt::RightDockWidgetArea);
    simDock->setWidget(simulationDialog);
    Application::getMainWindow()->addDockWidget(Qt::RightDockWidgetArea, simDock);
    simulationDialog->init();
    simDock->hide();
}

//----------------- destructor ----------------------------
LoadsManager::~LoadsManager() {
    // remove dock widget
    Application::getMainWindow()->removeDockWidget(simDock);
    // delete dialogs
    delete previewDialog;
    previewDialog = NULL;
    delete simulationDialog;
    simulationDialog = NULL;

    // reset color scale
    setAtomDataDisplay(NONE);

    // close current loads
    close();

    delete referencePM;
    referencePM = NULL;

}

//----------------- userWantsToSave ----------------------------
bool LoadsManager::userWantsToSave() {

    int exit = QMessageBox::warning(NULL, QMessageBox::tr("Save Changes?"),
                                    QMessageBox::tr("Loads have changed!\nIf you want to save, clic on Ok, otherwise cancel."),
                                    QMessageBox::Ok | QMessageBox::Default,
                                    QMessageBox::Cancel | QMessageBox::Escape);

    return (exit == QMessageBox::Ok);
}

//----------------- rewind ----------------------------
void LoadsManager::rewind() {
    // rewind everything
    if (previewDialog) {
        previewDialog->rewind();
        previewDialog->reset();
    }

    if (simulationDialog)
        simulationDialog->rewind();
}

//----------------- close ----------------------------
void LoadsManager::close() {
    // ask the user
    if (changed && userWantsToSave())
        save();

    changed = false;
    delete loadsEditor;
    loadsEditor = NULL;
    delete myLoads;
    myLoads = NULL;
    fileName = "";

    // hide 3D representation
    foreach(AtomDecoration *deco, representation3D) {
        // remove from the InteractiveViewer
        deco->show(false);
    }

//    if (showDisplacements)
//        myPMDC->destroyPointData();

    // refresh the view
    InteractiveViewer::get3DViewer()->refreshRenderer();
}

//----------------- deleteAllLoads ----------------------------
void LoadsManager::deleteAllLoads() {
    delete myLoads;
    myLoads = NULL;
    loadsChangedFlag = true;
}

//----------------- pause ----------------------------
void LoadsManager::pause() {
    if (previewDialog)
        previewDialog->pause();

    if (simulationDialog)
        simulationDialog->pause();
}

//----------------- addLoad ----------------------------
void LoadsManager::addLoad(Load *l) {
    if (!myLoads) {
        // we want to add a load, but there are no list defined yet
        myLoads = new Loads();
    }

    myLoads->addLoad(l);

    changed = true;
    loadsChangedFlag = true;
}

//----------------- open ----------------------------
bool LoadsManager::open(const QString & fileName) {
    if (myLoads != NULL)
        close();

    // set the current filename
    this->fileName = fileName;

    // open the loads stored in filename
    // and put everything in the list
    myLoads = new Loads(fileName.toStdString().c_str());

    changed = false;

    loadsChangedFlag = true;

    return (myLoads != NULL);
}


//----------------- save ----------------------------
void LoadsManager::save() {
    if (myLoads && myLoads->numberOfLoads()>0) {
        // test if the user already gave a filename
        if (fileName == "") {
            // open a file dialog to get a filename
            QString newFileName = QFileDialog::getSaveFileName(NULL, "Save LML As...", QFileInfo(fileName).absoluteDir().absolutePath(), "(*.lml)");

            if (!newFileName.isEmpty()) {
                // set the current fileName
                fileName = newFileName;
            } else {
                // tell the nice user that (s)he doesn't give a file name it will never
                // be able to save anything
                return;
            }
        }

        // never save to test.lml
        std::ofstream outputFile;
        outputFile.open(fileName.toStdString().c_str());
        myLoads->xmlPrint(outputFile);
        outputFile.close();

        changed = false;
    }
}

//----------------- saveAs ----------------------------
void LoadsManager::saveAs(const QString & fileName) {
    this->fileName = fileName;
    save();
}

//----------------- getLoads ----------------------------
Loads * LoadsManager::getLoads() {
    if (!myLoads)
        return NULL;

    return myLoads;
}

//----------------- setLoads ----------------------------
void LoadsManager::setLoads(Loads* loads) {
    deleteAllLoads();
    myLoads = loads;
}

//----------------- setDisplayLoads ----------------------------
void LoadsManager::setDisplayLoads(bool s) {
    if (showLoads && !s) {
        foreach(AtomDecoration * deco, representation3D) {
            deco->show(false);
        }
    }
    showLoads = s;
    updateLoadsDisplay();
    // refresh the view
    getPMManagerDC()->refresh();
    //InteractiveViewer::get3DViewer()->refreshRenderer();
}


//----------------- setAtomDataDisplay ----------------------------
void LoadsManager::setAtomDataDisplay(AtomDataDisplayType adt) {
    if (atomDataDisplay != NONE) {
        myPMManagerDC->destroyPointData();
    }

    atomDataDisplay = adt;

    QString scaleName;

    switch (atomDataDisplay) {
    case NONE:
        scaleName = "NONE! (if you see this message, congrats! you found a bug!)";
        break;
    case ADD_ON:
        scaleName = atomDataName;
        break;
    case DISPLACEMENTS:
        scaleName = "Displacements";
        break;
    case DISTANCES:
        scaleName = "Distances";
        break;
    case RELATIVE_ENERGY_NORM_ERROR:
        scaleName = "Relative Energy Norm Error (%)";
    }

    InteractiveViewer::get3DViewer()->setColorScale((atomDataDisplay != NONE));

    if (atomDataDisplay != NONE) {
        InteractiveViewer::get3DViewer()->setColorScaleTitle(scaleName);
        myPMManagerDC->createPointData();
    }

    // refresh the view
    // Application::getMainWindow()->getInteractiveViewer()->refresh();
    // force update
    if (previewDialog)
        previewDialog->updateDisplay(true);

    if (simulationDialog)
        simulationDialog->updateDisplay(true);
}


//----------------- setAtomData ----------------------------
void LoadsManager::setAtomData(std::AtomDataVector &pData, QString name) {
    atomData = pData;
    atomDataName = name;
    setAtomDataDisplay(ADD_ON);
}

//----------------- setReferencePML ----------------------------
void LoadsManager::setReferencePML(const QString & fileName) {
    referencePM = new PhysicalModel(fileName.toStdString().c_str());
}

//----------------- updateTime ----------------------------
void LoadsManager::updateTime() {
    if (simulationDialog)
        simulationDialog->updateTime(true);  // force the simulation dialog to get the motor add-on time
}

//---------------  updateAtomDataScale -----------------------
void LoadsManager::updateAtomDataScale(double min, double max) {
    this->min = min;
    this->max = max;

    if (!constrainedAtomDataScale)
        InteractiveViewer::get3DViewer()->setColorScaleMinMax(min, max);
}

//----------------- addDecoration ----------------------------
void LoadsManager::addDecoration(Atom *a, Load *ld, double time, double defaultSize, double *max, double *min, double *val) {
    AtomDecoration *deco = NULL;

    if (a) {
        if (ld->getType() == "Translation") {
            if (ld->getDirection().isXNull() && ld->getDirection().isYNull() && ld->getDirection().isZNull()) {
                // null translation
                deco = myPMManagerDC->getDC(a)->getDecoration("NullTranslation", GeometricObject::SPHERE);
                deco->setColor(0.4, 0.67, 0.88);
                deco->setSize(defaultSize);
            } else {
                // other translation
                deco = myPMManagerDC->getDC(a)->getDecoration("Translation", GeometricObject::ARROW);
                deco->setColor(0.6, 0.88, 0.59);
            }
        } else { // "Force", "Rotation", "Pressure"...
            deco = myPMManagerDC->getDC(a)->getDecoration("Force", GeometricObject::ARROW);
            deco->setColor(1.0, 0.3, 0.15);
        }
    }

    // update size and direction depending on the load value
    if (deco->getType() != GeometricObject::SPHERE) {
        double pos2[3];
        a->getPosition(pos2);

        if (ld->getDirection().isToward()) {
            int twdind = ld->getDirection().getToward();
            Atom * twd = myPMManagerDC->getPhysicalModel()->getAtom(twdind);

            if (twd) {
                double pos1[3];
                twd->getPosition(pos1);
                deco->update(pos1[0] - pos2[0], pos1[1] - pos2[1], pos1[2] - pos2[2]);
            }
        } else {
            double x, y, z;
            ld->getDirection(x, y , z);
            deco->update(x, y, z);
        }

        *val = ld->getValue(time);

        if (*max < *val)
            *max = *val;

        if (*min > *val)
            *min = *val;
    }

    // show in scene
    deco->show(true);

    // store in case we need to hide all
    representation3D.insert(deco);

}


//----------------- updateLoadsDisplay ----------------------------
void LoadsManager::updateLoadsDisplay() {
    if (showLoads) {
        PhysicalModel * myPM = myPMManagerDC->getPhysicalModel();
        if (myLoads != NULL && myPM != NULL) {
            // for all loads, display the decoration
            Load * ld;

            // ask the preview dialog or simulation dialog: "what time is it?"
            double time = (previewDialog) ? previewDialog->getTime() :
                          ((simulationDialog) ? simulationDialog->getTime() : 0.0);

            // default size for sphere decoration
            double defaultSize = myPMManagerDC->getBoundingRadius() / 15.0;

            // max value of active loads
            double max = -1.0, min = 99999999.9 /*DBL_MAX*/;
            std::vector <double> loadsValue;
            double val;

            for (unsigned i = 0; i < myLoads->numberOfLoads(); i++) {
                ld = myLoads->getLoad(i);

                if (ld->isActive(time)) {

                    for (unsigned j = 0; j < ld->numberOfTargets(); j++) {

                        if (ld->getTargetList().indexedTargets()) {
                            // get the atom ptr from its index
                            addDecoration(myPM->getAtom(ld->getTarget(j)), ld, time, defaultSize, &max, &min, &val);
                            loadsValue.push_back(val);
                        } else {
                            // get all atoms from the named target
                            std::string targetComponent = ld->getTargetList().getNamedTarget(j);

                            // look for the component in the informative part of the physical model
                            ::Component * c = myPM->getInformativeComponents()->getComponentByName(targetComponent);

                            if (c != NULL) {
                                for (unsigned int cellId = 0; cellId < c->getNumberOfCells(); cellId++) {
                                    Cell *cell = c->getCell(cellId);

                                    for (unsigned int atomId = 0; atomId < cell->getNumberOfStructures(); atomId++) {
                                        addDecoration((Atom*) cell->getStructure(atomId), ld, time, defaultSize, &max, &min, &val);
                                        loadsValue.push_back(val);
                                    }
                                }
                            }
                        }
                    }
                }
            }

            // resize the arrows
            double size;
            double maxSize = defaultSize * 5.0;
            double minSize = defaultSize;
            unsigned int j = 0;
            foreach(AtomDecoration * deco, representation3D) {
                if (deco->getType() != GeometricObject::SPHERE) {
                    val = loadsValue[j++];

                    if (fabs(max - min) < 1e-10 || fabs(max - val) < 1e-10) {
                        size = maxSize;
                    } else {
                        if (fabs(min - val) < 1e-10) {
                            size = minSize;
                        } else {
                            // renormalized the size between min and max
                            size = ((val - min) / (max - min)) * (maxSize - minSize) + minSize;
                        }
                    }

                    deco->setSize(size);
                }
            }

        }
    }

}

//----------------- editLoads ----------------------------
void LoadsManager::editLoads() {
    // open the load editor dialog
    if (!loadsEditor)
        loadsEditor = new LoadsEditor(this, NULL);
    else {
        loadsEditor->updateLoads();
    }

    // show it
    loadsEditor->show();

    loadsChangedFlag = true;
}

//----------------- previewLoads ----------------------------
void LoadsManager::previewLoads() {
    if (!previewDialog)
        previewDialog = new LoadsMovie(this, NULL);

    previewDialog->show();
}

//----------------- addLoad ----------------------------
void LoadsManager::addLoad() {
    // here selectedDC is a vector that contains all the selected DCs
    std::vector<camitk::Component *> selectedDC;

    foreach(camitk::Component *dc, camitk::Application::getSelectedComponents()) {
        if (dc->isInstanceOf("AtomDC"))
            selectedDC.push_back(dc);
    }

    LoadsEditor * addDialog = new LoadsEditor(selectedDC, this, NULL);

    // show it
    addDialog->show();
    loadsChangedFlag = true;

}


//----------------- simulation ----------------------------
void LoadsManager::simulation() {
    if (!simDock->isVisible())
        simDock->show();
    else
        simDock->hide();
}

//----------------- addSimulationTab ----------------------------
void LoadsManager::addSimulationTab(QWidget*w) {
    simulationDialog->show();
    simulationDialog->addTab(w);
}

//----------------- getAnimationMotorAddonLocation ----------------------------
QString LoadsManager::getAnimationMotorAddonLocation() {
    return motorAddonLocation;
}

//----------------- setAnimationMotorAddonLocation ----------------------------
void LoadsManager::setAnimationMotorAddonLocation(QString mloc) {
    motorAddonLocation = mloc;
}

//----------------- setAnimationMotorAddonLocation ----------------------------
void LoadsManager::userConstrainedAtomDataScale ( bool constrained) {
    constrainedAtomDataScale = constrained;
}

//----------------- setAnimationMotorAddonLocation ----------------------------
bool LoadsManager::getUserConstrainedAtomDataScale() {
    return constrainedAtomDataScale;
}

//----------------- setAnimationMotorAddonLocation ----------------------------
void  LoadsManager::getAtomDataScale ( double* min , double* max) {
    *min = this->min;
    *max = this->max;
}
