package charactermanaj.model.util;

import java.io.File;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;

import charactermanaj.model.AppConfig;
import charactermanaj.model.io.CharacterDataPersistent;
import charactermanaj.util.DirectoryConfig;
import charactermanaj.util.UserDataFactory;


/**
 * 開始前の事前準備するためのサポートクラス 
 * @author seraphy
 */
public abstract class StartupSupport {
	
	private static StartupSupport inst;
	
	/**
	 * インスタンスを取得する.
	 * @return シングルトンインスタンス
	 */
	public static synchronized StartupSupport getInstance() {
		if (inst == null) {
			inst = new StartupSupport() {
				private final Logger logger = Logger.getLogger(StartupSupport.class.getName());

				@Override
				public void doStartup() {
					StartupSupport[] startups = {
							new PurgeOldLogs(),
							new UpgradeCache(),
							new UpgradeFavoritesXml(),
							new PurgeUnusedCache(),
					};
					for (StartupSupport startup : startups) {
						logger.log(Level.FINE, "startup operation start. class="
								+ startup.getClass().getSimpleName());
						try {
							startup.doStartup();
							logger.log(Level.FINE, "startup operation is done.");

						} catch (Exception ex) {
							logger.log(Level.WARNING, "startup operation failed.", ex);
						}
					}
				}
			};
		}
		return inst;
	}
	
	/**
	 * ユーザディレクトリをアップグレードします.
	 */
	public abstract void doStartup();
}

/**
 * 古い形式のCacheを移動する.
 * @author seraphy
 *
 */
class UpgradeCache extends StartupSupport {
	
	/**
	 * ロガー
	 */
	private final Logger logger = Logger.getLogger(getClass().getName());

	@Override
	public void doStartup() {
		UserDataFactory userDataFactory = UserDataFactory.getInstance();
		File appData = userDataFactory.getSpecialDataDir(null);

		// ver0.94まではユーザディレクトリ直下に*.serファイルを配置していたが、
		// ver0.95以降はcachesに移動したため、旧ファイルがのこっていれば移動する.
		for (File file : appData.listFiles()) {
			try {
				String name = file.getName();
				if (file.isFile() && name.endsWith(".ser")) {
					File toDir = userDataFactory.getSpecialDataDir(name);
					if ( !appData.equals(toDir)) {
						String convertedName = convertName(name);
						File toFile = new File(toDir, convertedName);
						boolean ret = file.renameTo(toFile);
						logger.log(Level.INFO, "move file " + file + " to " + toFile + " ;successed=" + ret);
					}
				}
			} catch (Exception ex) {
				logger.log(Level.WARNING, "file move failed. " + file, ex);
			}
		}
	}
	
	protected String convertName(String name) {
		if (name.endsWith("-workingset.ser") || name.endsWith("-favorites.ser")) {
			// {UUID}-CharacterID-workingset.ser もしくは、-favorites.serの場合は、
			// {UUID}-workingset.ser / {UUID}-favorites.serに変更する.
			String[] splitedName = name.split("-");
			if (splitedName.length >= 7) {
				StringBuilder buf = new StringBuilder();
				for (int idx = 0; idx < 5; idx++) {
					if (idx > 0) {
						buf.append("-");
					}
					buf.append(splitedName[idx]);
				}
				buf.append("-");
				buf.append(splitedName[splitedName.length - 1]);
				return buf.toString();
			}
		}
		return name;
	}
}

/**
 * DocBaseのハッシュ値をもとにしたデータをサポートする抽象クラス.
 * @author seraphy
 *
 */
abstract class StartupSupportForDocBasedData extends StartupSupport {
	
	/**
	 * character.xmlのファイル位置を示すUUID表現を算定するためのアルゴリズムの選択肢.<br>
	 * @author seraphy
	 */
	protected enum DocBaseSignatureStoratage {
		
		/**
		 * 新形式のcharacter.xmlのUUIDを取得(もしくは生成)する.
		 * charatcer.xmlファイルのURIを文字列にしたもののタイプ3-UUID表現.<br>
		 */
		NEW_FORMAT() {
			@Override
			public Map<File, String> getDocBaseSignature(Collection<File> characterXmlFiles) {
				HashMap<URI, File> uris = new HashMap<URI, File>();
				for (File characterXmlFile : characterXmlFiles) {
					uris.put(characterXmlFile.toURI(), characterXmlFile);
				}
				UserDataFactory userDataFactory = UserDataFactory.getInstance();
				HashMap<File, String> results = new HashMap<File, String>();
				File storeDir = userDataFactory.getSpecialDataDir("*.ser");
				for (Map.Entry<URI, String> entry : userDataFactory
						.getMangledNameMap(uris.keySet(), storeDir, true).entrySet()) {
					File characterXmlFile = uris.get(entry.getKey());
					String mangledName = entry.getValue();
					results.put(characterXmlFile, mangledName);
				}
				return results;
			}
		},
		
		/**
		 * 旧形式のcharacter.xmlのUUIDを取得する.<br>
		 * charatcer.xmlファイルのURLを文字列にしたもののタイプ3-UUID表現.<br>
		 */
		OLD_FORMAT() {
			@Override
			public Map<File, String> getDocBaseSignature(Collection<File> characterXmlFiles) {
				HashMap<File, String> results = new HashMap<File, String>();
				for (File characterXmlFile : characterXmlFiles) {
					String mangledName;
					try {
						@SuppressWarnings("deprecation")
						URL url = characterXmlFile.toURL();
						mangledName = UUID.nameUUIDFromBytes(url.toString().getBytes()).toString();

					} catch (Exception ex) {
						logger.log(Level.WARNING,
								"character.xmlのファイル位置をUUID化できません。:"
										+ characterXmlFile, ex);
						mangledName = null;
					}
					results.put(characterXmlFile, mangledName);
				}
				return results;
			}
		},
		;
		
		/**
		 * ロガー
		 */
		private static Logger logger = Logger.getLogger(
				DocBaseSignatureStoratage.class.getName());
		
		/**
		 * character.xmlからuuid表現のプレフィックスを算定する.
		 * @param characterXmlFile character.xmlのファイルのコレクション
		 * @return character.xmlファイルと、それに対応するUUIDのマップ、(UUIDは該当がなければnull)
		 */
		public abstract Map<File, String> getDocBaseSignature(Collection<File> characterXmlFiles);
	}
	
	/**
	 * すべてのユーザおよびシステムのキャラクターデータのDocBaseをもととした、
	 * キャッシュディレクトリ上のハッシュ値(Prefix)の文字列をキーとし、
	 * そのキャラクターディレクトリを値とするマップを返す.<br>
	 * (新タイプの場合は実在するcharacter.xmlに対するmangledNameが生成され登録される.)<br>
	 * @param storatage ハッシュ値を生成する戦略(旧タイプ・新タイプのUUIDの区別のため)
	 * @return DocBaseをもととしたハッシュ値の文字列表記をキー、キャラクターディレクトリを値とするマップ
	 */
	protected Map<String, File> getDocBaseMapInCaches(DocBaseSignatureStoratage storatage) {
		if (storatage == null) {
			throw new IllegalArgumentException();
		}

		// キャラクターデータフォルダ
		DirectoryConfig dirConfig = DirectoryConfig.getInstance();
		File[] charactersDirs = {
				dirConfig.getCharactersDir() // 現在のキャラクターデータフォルダ
		};
		
		// キャラクターデータディレクトリを走査し、character.xmlファイルを収集する.
		ArrayList<File> characterXmlFiles = new ArrayList<File>();
		for (File charactersDir : charactersDirs) {
			if (charactersDir == null || !charactersDir.exists()
					|| !charactersDir.isDirectory()) {
				continue;
			}
			for (File characterDir : charactersDir.listFiles()) {
				if ( !characterDir.isDirectory()) {
					continue;
				}
				File characterXml = new File(characterDir, CharacterDataPersistent.CONFIG_FILE);
				if ( !characterXml.exists()) {
					continue;
				}
				characterXmlFiles.add(characterXml);
			}
		}

		// character.xmlファイルに対するハッシュ化文字列を取得する.
		Map<File, String> docBaseSigMap = storatage.getDocBaseSignature(characterXmlFiles);
		
		// ハッシュ化文字列をキーとし、そのcharacter.xmlファイルを値とするマップに変換する.
		HashMap<String, File> docBaseSignatures = new HashMap<String, File>();
		for (Map.Entry<File, String> entry : docBaseSigMap.entrySet()) {
			docBaseSignatures.put(entry.getValue(), entry.getKey());
		}
		return docBaseSignatures;
	}
	
	/**
	 * 指定したディレクトリ直下にあるDocBaseのUUIDと推定される文字列で始まり、suffixで終わる
	 * ファイルのUUIDの部分文字列をキーとし、そのファイルを値とするマップを返す.
	 * @param dataDir 対象ディレクトリ
	 * @param suffix  対象となるファイル名の末尾文字列、nullまたは空文字の場合は全て
	 * @return DocBaseのUUID表現と思われる部分文字列をキーとし、そのファイルを値とするマップ
	 */
	protected Map<String, File> getUUIDMangledNamedMap(File dataDir, String suffix) {
		if (dataDir == null || !dataDir.exists() || !dataDir.isDirectory()) {
			throw new IllegalArgumentException();
		}

		Map<String, File> uuidMangledFiles = new HashMap<String, File>();
		for (File file : dataDir.listFiles()) {
			String name = file.getName();
			if (file.isFile() && (suffix == null || suffix.length() == 0 || name.endsWith(suffix))) {
				String[] sigParts = name.split("-");
				if (sigParts.length >= 5) {
					// UUIDはa-b-c-d-eの5パーツになり、更に末尾に何らかの文字列が付与されるので
					// 区切り文字は5つ以上になる.
					// UUIDの部分だけ結合し直して復元する.
					StringBuilder sig = new StringBuilder();
					for (int idx = 0; idx < 5; idx++) {
						if (idx != 0) {
							sig.append("-");
						}
						sig.append(sigParts[idx]);
					}
					uuidMangledFiles.put(sig.toString(), file);
				}
			}
		}
		return uuidMangledFiles;
	}
	
}

/**
 * 古い形式のFavorites.xmlを移動する.
 * @author seraphy
 */
class UpgradeFavoritesXml extends StartupSupportForDocBasedData {
	
	/**
	 * ロガー
	 */
	private final Logger logger = Logger.getLogger(getClass().getName());

	@Override
	public void doStartup() {
		UserDataFactory userDataFactory = UserDataFactory.getInstance();
		File appData = userDataFactory.getSpecialDataDir(null);

		// キャラクターデータディレクトリを走査しdocBaseの識別子の一覧を取得する
		Map<String, File> docBaseSignatures = getDocBaseMapInCaches(
				DocBaseSignatureStoratage.OLD_FORMAT);

		// ver0.94までは*.favorite.xmlはユーザディレクトリ直下に配備していたが
		// ver0.95以降は各キャラクターディレクトリに移動するため、旧docbase-id-favorites.xmlが残っていれば移動する
		// ユーザディレクトリ直下にある*-facotites.xmlを列挙する
		Map<String, File> favorites = getUUIDMangledNamedMap(appData, "-favorites.xml");

		// 旧式の*-favorites.xmlを各キャラクターディレクトリに移動する.
		for (Map.Entry<String, File> favoritesEntry : favorites.entrySet()) {
			String sig = favoritesEntry.getKey();
			File file = favoritesEntry.getValue();
			try {
				File characterDir = docBaseSignatures.get(sig);
				if (characterDir != null) {
					File toFile = new File(characterDir, "favorites.xml");
					boolean ret = file.renameTo(toFile);
					logger.log(Level.INFO, "move file " + file + " to " + toFile + " ;successed=" + ret);
				}

			} catch (Exception ex) {
				logger.log(Level.INFO, "move file failed." + file, ex);
			}
		}
	}
}

/**
 * 存在しないDocBaseを参照するキャッシュ(*.ser)を削除する.<br>
 * (キャラクターディレクトリを変更した場合は、以前のデータは削除される.)<br>
 * @author seraphy
 */
class PurgeUnusedCache extends StartupSupportForDocBasedData {
	
	/**
	 * ロガー
	 */
	private final Logger logger = Logger.getLogger(getClass().getName());

	@Override
	public void doStartup() {
		// キャッシュの保存先を取得する.
		UserDataFactory userDataFactory = UserDataFactory.getInstance();
		File cacheDir = userDataFactory.getSpecialDataDir("*.ser");
		
		// 現在選択されているキャラクターデータディレクトリ上のUUIDの登録を保証する.
		// (character.xmlに対するUUIDの問い合わせ時に登録がなければ登録される.)
		// (現在選択されていないディレクトリについてはUUIDの登録が行われないので、もしデータベースファイルが
		// 作成されていない場合はキャッシュは一旦削除されることになる.)
		getDocBaseMapInCaches(DocBaseSignatureStoratage.NEW_FORMAT);
		
		// キャッシュ上にあるDocBaseのUUID表現で始まる*.serを列挙
		String[] suffixes = {
				"-character.xml-cache.ser", // character.xmlのキャッシュ
				"-workingset.ser", // 作業状態のキャッシュ
				"-favorites.ser" // お気に入りのキャッシュ
				};

		// UUIDからURIの索引
		final Map<String, URI> mangledURIMap = userDataFactory.getMangledNameMap(cacheDir);

		// キャッシュファイルで使用されているUUIDの示す実体のURIが実在しない場合は、そのキャッシュは不要と見なす.
		for (String suffix : suffixes) {
			Map<String, File> caches = getUUIDMangledNamedMap(cacheDir, suffix);
			for (Map.Entry<String, File> cacheEntry : caches.entrySet()) {
				String mangledUUID = cacheEntry.getKey();
				File cacheFile = cacheEntry.getValue();
				try {
					URI uri = mangledURIMap.get(mangledUUID);
					boolean remove = true;
					if (uri != null) {
						File characterXmlFile = new File(uri);
						if (characterXmlFile.exists() || characterXmlFile.isFile()) {
							// UUIDデータベースに登録があり、且つ、
							// character.xmlが実在する場合のみ削除しない.
							remove = false;
						}
					}
					if (remove) {
						// キャッシュファイルを削除する.
						boolean result = cacheFile.delete();
						logger.log(Level.INFO, "purge unused cache: " + cacheFile
								+ "/succeeded=" + result);
					}
					
				} catch (Exception ex) {
					logger.log(Level.WARNING, "remove file failed. " + cacheFile, ex);
				}
			}
		}
	}
}

/**
 * 古いログファイルを消去する.
 * @author seraphy
 */
class PurgeOldLogs extends StartupSupport {

	/**
	 * ロガー
	 */
	private final Logger logger = Logger.getLogger(getClass().getName());

	@Override
	public void doStartup() {
		UserDataFactory userDataFactory = UserDataFactory.getInstance();
		File logsDir = userDataFactory.getSpecialDataDir("*.log");
		if (logsDir.exists()) {
			AppConfig appConfig = AppConfig.getInstance();
			long purgeOldLogsMillSec = appConfig.getPurgeLogDays() * 24L * 3600L * 1000L;
			if (purgeOldLogsMillSec > 0) {
				long purgeThresold = System.currentTimeMillis() - purgeOldLogsMillSec;
				for (File file : logsDir.listFiles()) {
					try {
						String name = file.getName();
						if (file.isFile() && file.canWrite() && name.endsWith(".log")) {
							long lastModified = file.lastModified();
							if (lastModified > 0 && lastModified < purgeThresold) {
								boolean result = file.delete();
								logger.log(Level.INFO, "remove file " + file
										+ "/succeeded=" + result);
							}
						}

					} catch (Exception ex) {
						logger.log(Level.WARNING, "remove file failed. " + file, ex);
					}
				}
			}
		}
	}
}
