/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file callmonitor.c
 * \brief Contains all function needed for callmonitor window
 */

#include <ffgtk.h>

enum {
	COL_CM_TYPE,
	COL_CM_DATETIME,
	COL_CM_NAME,
	COL_CM_COMPANY,
	COL_CM_CITY,
	COL_CM_NUMBER,
	COL_CM_LOCALNAME,
	COL_CM_LOCALNUMBER,
	COL_CM_DURATION
};

/** Common used widgets */
static gchar *pnCallerList = NULL;
/** Call monitor window width */
static gint nCallMonitorWidth = 0;
/** Call monitor window height */
static gint nCallMonitorHeight = 0;
/** Call monitor widget */
static GtkWidget *psCallMonitor = NULL;
/** Call monitor list store */
static GtkListStore *psMonitorStore = NULL;
/** Filter list store */
static GtkListStore *psFilterStore = NULL;
/** list model */
static GtkTreeModel *psListStore = NULL;
/** Selection treeview */
static GtkWidget *psSelectionTreeView = NULL;
/** call monitor treeview */
static GtkWidget *psCallMonitorTreeView = NULL;
/** Pane press position */
static int nPressPosition = -1;
/** lookup combo box */
static GtkWidget *psLookupComboBox = NULL;
/** toolbar */
static GtkWidget *psToolbar = NULL;
/** Display all profiles flag */
static gboolean bAllProfiles = FALSE;

/** type column */
static GtkTreeViewColumn *psTypeColumn = NULL;
/** date/time column */
static GtkTreeViewColumn *psDateTimeColumn = NULL;
/** name column */
static GtkTreeViewColumn *psNameColumn = NULL;
/** company column */
static GtkTreeViewColumn *psCompanyColumn = NULL;
/** city column */
static GtkTreeViewColumn *psCityColumn = NULL;
/** number column */
static GtkTreeViewColumn *psNumberColumn = NULL;
/** local name column */
static GtkTreeViewColumn *psLocalNameColumn = NULL;
/** local number column */
static GtkTreeViewColumn *psLocalNumberColumn = NULL;
/** duration column */
static GtkTreeViewColumn *psDurationColumn = NULL;
/** Current displayed profile */
static struct sProfile *psCallMonitorProfile = NULL;
/** Current selected filter */
struct sFilter *psCurrentFilter = NULL;
/** indicates if external update threads are working */
static gchar nWorkExt = 0;

static gboolean bFinished = FALSE;

/**
 * \brief Get caption text depending on selected profile(s)
 * \return caption string
 */
static gchar *getProfileCaption( void ) {
	if ( bAllProfiles == TRUE ) {
		return g_strdup( _( "Call Monitor (all profiles)" ) );
	}

	if ( psCallMonitorProfile == NULL ) {
		return g_strdup_printf( _( "Call Monitor (%s)" ), getActiveProfile() -> pnName );
	}

	return g_strdup_printf( _( "Call Monitor (%s)" ), psCallMonitorProfile -> pnName );
}

/**
 * \brief Get caller list
 * \param psProfile profile pointer
 * \return caller list
 */
gchar *getCallerList( struct sProfile *psProfile ) {
	if ( pnCallerList != NULL ) {
		g_free( pnCallerList );
	}

	pnCallerList = g_strdup_printf( "%s/%s/callerlist.csv", getProfilesDir(), psProfile -> pnName );

	return pnCallerList;
}

/**
 * \brief Free caller list and set to null
 */
void freeCallerList( void ) {
	if ( pnCallerList != NULL ) {
		g_free( pnCallerList );
		pnCallerList = NULL;
	}
}

/**
 * \brief Get filter info
 * \param psFilter current filter structure
 * \param psInfo filter info
 */
void getFilterInfo( struct sFilter *psFilter, struct sFilterInfo *psInfo ) {
	GList *psList;
	gint nCount = 0;
	gint nDuration = 0;

	for ( psList = psCallerList; psList != NULL; psList = psList -> next ) {
		struct sCaller *psCaller = psList -> data;

		if ( psCaller != NULL ) {
			if ( checkFilter( psCaller, psFilter ) ) {
				continue;
			}

			if ( strchr( psCaller -> pnDuration, 's' ) != NULL ) {
				/* Ignore voicebox duration */
			} else {
				if ( psCaller -> pnDuration != NULL && strlen( psCaller -> pnDuration ) > 0 ) {
					nDuration += ( psCaller -> pnDuration[ 0 ] - '0' ) * 60;
					nDuration += ( psCaller -> pnDuration[ 2 ] - '0' ) * 10;
					nDuration += psCaller -> pnDuration[ 3 ] - '0';
				}
			}
			nCount++;
		}
	}

	if ( psInfo != NULL ) {
		psInfo -> nCount = nCount;
		psInfo -> nDuration = nDuration;
	}
}

/**
 * \brief Read fax and voice box and add to list
 * \param pUserData UNUSED
 * \return NULL
 */
static gpointer updateFaxVoiceBox( gpointer pUserData ) {
	routerLoadFaxBox( getActiveProfile() );
	routerLoadVoiceBox( getActiveProfile() );

	/* Update filter list */
	ffgtkLock();
	gtk_list_store_clear( psFilterStore );
	updateFilter( psFilterStore, psSelectionTreeView );
	Debug( KERN_DEBUG, "done\n" );
	ffgtkUnlock();

	nWorkExt--;

	return NULL;
}

/**
 * \brief Load call list from file
 */
static void loadCallerList( void ) {
	struct sProfile *psProfile = NULL;
	GList *psList;
	gchar *pnDummy = NULL;

	while ( nWorkExt != 0 ) {
		GTK_FLUSH;
	}
	nWorkExt = 1;

	if ( bAllProfiles == TRUE ) {
		for ( psList = getProfiles(); psList != NULL; psList = psList -> next ) {
			psProfile = psList -> data;

			if ( psProfile != NULL ) {
				pnDummy = g_strdup_printf( "%s/%s/callerlist.csv", getProfilesDir(), psProfile -> pnName );
				parseCsvFile( pnDummy, psProfile -> pnName );
				g_free( pnDummy );
			}
		}
	} else {
		parseCsvFile( getCallerList( psCallMonitorProfile ), psCallMonitorProfile -> pnName );
	}

	if ( bAllProfiles == TRUE || psCallMonitorProfile == getActiveProfile() ) {
		nWorkExt++;
		CREATE_THREAD( "faxvoicebox", updateFaxVoiceBox, NULL );
	}

	nWorkExt--;
}

/**
 * \brief Add caller to list store
 * \param psCaller caller structure
 * \param psPrivStore list store pointer or NULL
 * \param psPrivFilter filter pointer or NULL
 * \return call duration
 */
gint callMonitorAddCaller( struct sCaller *psCaller, GtkListStore *psPrivStore, struct sFilter *psPrivFilter ) {
	GtkListStore *psStore = psPrivStore != NULL ? psPrivStore : GTK_LIST_STORE( psListStore );
	struct sFilter *psFilter = psPrivFilter != NULL ? psPrivFilter : psCurrentFilter;
	GtkTreeIter sIter;
	gchar *pnTmp;
	gint nTotalDuration = 0;

	/* only add entries which applies to the current filter rule */
	if ( checkFilter( psCaller, psFilter ) ) {
		return nTotalDuration;
	}

	/* Add list store entry */
	gtk_list_store_append( psStore, &sIter );

	/* Set caller type icon */
	gtk_list_store_set( psStore, &sIter, COL_CM_TYPE, getTypeIcon( psCaller -> nType ), -1 );

	/* Set date/time */
	pnTmp = g_locale_to_utf8( psCaller -> pnDateTime, -1, 0, 0, 0 );
	gtk_list_store_set( psStore, &sIter, COL_CM_DATETIME, pnTmp, -1 );
	g_free( pnTmp );

	/* Add name:
	 * First check if we can find the number within our addressbook
	 * Second add name if we already have one
	 * Third add empty string
	 */
	struct sPerson *psPerson = findPersonByNumber( psCaller -> pnNumber );
	if ( psPerson != NULL ) {
		pnTmp = g_strdup( psPerson -> pnDisplayName );
	} else if ( psCaller -> pnName != NULL && strlen( psCaller -> pnName ) > 0 ) {
		pnTmp = convert_utf8( psCaller -> pnName );
	} else {
		pnTmp = g_locale_to_utf8( "", -1, 0, 0, 0 );
	}
	gtk_list_store_set( psStore, &sIter, COL_CM_NAME, pnTmp, -1 );
	g_free( pnTmp );

	/* Add company information if we have one, else none */
	if ( psPerson != NULL ) {
		pnTmp = convert_utf8( psPerson -> pnCompany );
	} else {
		pnTmp = g_locale_to_utf8( "", -1, 0, 0, 0 );
	}
	gtk_list_store_set( psStore, &sIter, COL_CM_COMPANY, pnTmp, -1 );
	g_free( pnTmp );

	/* Add the full number */
	gchar *pnFull = NULL;
	if ( isdigit( psCaller -> pnNumber[ 0 ] ) ) { 
		pnFull = fullNumber( psCaller -> pnNumber, FALSE );
		if ( pnFull != NULL ) {
			pnTmp = g_locale_to_utf8( pnFull, -1, 0, 0, 0 );
			g_free( pnFull );
		} else {
			pnTmp = g_locale_to_utf8( "", -1, 0, 0, 0 );
		}
	} else {
		pnTmp = g_locale_to_utf8( psCaller -> pnNumber, -1, 0, 0, 0 );
	}
	gtk_list_store_set( psStore, &sIter, COL_CM_NUMBER, pnTmp, -1 );
	g_free( pnTmp );

	/* Add city information */
	pnFull = fullNumber( psCaller -> pnNumber, TRUE );
	if ( pnFull != NULL ) {
		gchar *pnCity = getCityByNumber( pnFull );

		if ( pnCity != NULL ) {
			pnTmp = convert_utf8( pnCity );
		} else {
			pnTmp = g_locale_to_utf8( "", -1, 0, 0, 0 );
		}
		g_free( pnFull );
	} else {
		pnTmp = g_locale_to_utf8( "", -1, 0, 0, 0 );
	}
	gtk_list_store_set( psStore, &sIter, COL_CM_CITY, pnTmp, -1 );
	g_free( pnTmp );

	/* Set local name */
	pnTmp = convert_utf8( psCaller -> pnLocalName );
	gtk_list_store_set( psStore, &sIter, COL_CM_LOCALNAME, pnTmp, -1 );
	g_free( pnTmp );

	/* Set local number */
	pnTmp = g_locale_to_utf8( psCaller -> pnLocalNumber, -1, 0, 0, 0 );
	gtk_list_store_set( psStore, &sIter, COL_CM_LOCALNUMBER, pnTmp, -1 );
	g_free( pnTmp );

	/* Set duration */
	pnTmp = g_locale_to_utf8( psCaller -> pnDuration, -1, 0, 0, 0 );
	gtk_list_store_set( psStore, &sIter, COL_CM_DURATION, pnTmp, -1 );

	nTotalDuration += 60 * atol( pnTmp );
	nTotalDuration += atol( pnTmp + strlen( pnTmp ) - 2 );
	
	g_free( pnTmp );

	return nTotalDuration;
}

/**
 * \brief Load call list from file
 * \param psStore list store where we add the information
 * \param psFilter current filter structure
 * \return total duration time
 */
static gint displayCallerList( GtkListStore *psStore, struct sFilter *psFilter ) {
	GList *psList = NULL;
	gint nTotalDuration = 0;

	/* loop through caller list and entries to list store */
	for ( psList = psCallerList; psList != NULL; psList = psList -> next ) {
		struct sCaller *psCaller = psList -> data;

		/* Check for a valid caller entry */
		if ( psCaller != NULL ) {
			nTotalDuration += callMonitorAddCaller( psCaller, psStore, psFilter );
		}
	}

	return nTotalDuration;
}

/**
 * \brief Move progress bar
 * \param pData user data pointer
 * \return TRUE (do not delete timer)
 */
gboolean updateProgressBar( gpointer pData ) {
	gtk_progress_bar_pulse( GTK_PROGRESS_BAR( pData ) );

	return TRUE;
}

/**
 * \brief Handle reload button call
 * \param pData list store pointer
 * \return NULL
 */
static gpointer reloadCallMonitor( gpointer pData ) {
	gint nError = 0;

	nError = routerGetCallList( getActiveProfile() );
	if ( nError != 0 ) {
		if ( nError == 33485 ) {
			ShowError( FALSE, _( "Login blocked" ), _( "ffgtk has blocked login until valid password is given" ) );
		} else if ( nError < 0 ) {
			RouterLoginError( FALSE, -nError );
		} else {
			ShowError( FALSE, _( "Network error" ), curl_easy_strerror( nError ) );
		}
	}

	bFinished = TRUE;

	return NULL;
}

/**
 * \brief Handle reload button call
 * \param psWidget button widget
 * \param pData list store pointer
 */
static void reloadCallMonitorStart( GtkWidget *psWidget, gpointer pData ) {
	GtkListStore *psListStore = GTK_LIST_STORE( pData );

	GdkCursor *psCursor = NULL;
	bFinished = FALSE;
	psCursor = gdk_cursor_new_for_display( gtk_widget_get_display( psCallMonitor ), GDK_WATCH );
	gdk_window_set_cursor( gtk_widget_get_window( GTK_WIDGET( psCallMonitor ) ), psCursor );
	if ( psCursor != NULL ) {
#if GTK_MAJOR_VERSION < 3
		gdk_cursor_unref( psCursor );
#else
		g_object_unref( psCursor );
#endif
	}

	GTK_FLUSH;

	if ( callMonitorGetReplaceOnLoad( getActiveProfile() ) == TRUE ) {
		freeCallers();
	}

	CREATE_THREAD( "callmonitor", reloadCallMonitor, NULL );

	while ( bFinished == FALSE ) usleep( 10 );
	gtk_list_store_clear( psListStore );
	loadCallerList();
	displayCallerList( psListStore, psCurrentFilter );
	gdk_window_set_cursor( gtk_widget_get_window( GTK_WIDGET( psCallMonitor ) ), NULL );
}

/**
 * \brief Display all profiles caller lists in call monitor
 * \param psWidget UNUSED
 * \param pData UNUSED
 */
static void allProfiles( GtkWidget *psWidget, gpointer pData ) {
	gchar *pnDummy = NULL;

	bAllProfiles = TRUE;

	pnDummy = getProfileCaption();
	gtk_window_set_title( GTK_WINDOW( psCallMonitor ), pnDummy );
	g_free( pnDummy );

	freeCallers();

	gtk_list_store_clear( psMonitorStore );
	loadCallerList();
	displayCallerList( psMonitorStore, psCurrentFilter );
}

/**
 * \brief Handle next profile button call
 * \param psWidget button widget
 * \param pData user data pointer
 */
static void nextProfile( GtkWidget *psWidget, gpointer pData ) {
	GList *psList = NULL;
	struct sProfile *psProfile = NULL;
	gchar *pnDummy = NULL;

	/* Set all profiles flag to FALSE */
	bAllProfiles = FALSE;

	/* Get profiles list */
	psList = getProfiles();

	/* In case we do not have an entry, return */
	if ( g_list_length( psList ) <= 0 ) {
		return;
	}

	/* Iterate through list and try to find last call monitor profile */
	for ( psList = getProfiles(); psList != NULL; psList = psList -> next ) {
		psProfile = psList -> data;

		if ( psProfile != NULL && psProfile == psCallMonitorProfile ) {
			psProfile = NULL;
			psList = psList -> next;
			if ( psList != NULL ) {
				/* got next entry */
				psProfile = psList -> data;
			}
			break;
		}
	}

	/* In case profile is still NULL, start with the first one in list */
	if ( psProfile == NULL ) {
		psProfile = getProfiles() -> data;
	}
	psCallMonitorProfile = psProfile;

	/* Update window caption */
	pnDummy = getProfileCaption();
	gtk_window_set_title( GTK_WINDOW( psCallMonitor ), pnDummy );
	g_free( pnDummy );

	/* Clear caller structure */
	freeCallers();

	/* clear list store and add current caller list */
	gtk_list_store_clear( psMonitorStore );
	loadCallerList();
	displayCallerList( psMonitorStore, psCurrentFilter );

	/* Update filter list */
	gtk_list_store_clear( psFilterStore );
	updateFilter( psFilterStore, psSelectionTreeView );
}

/**
 * \brief Handle delete button call
 * \param psWidget button widget
 * \param pData user data pointer
 */
static void deleteCallMonitor( GtkWidget *psWidget, void *pData ) {
	GtkWidget *psDialog = NULL;
	gint nResult = 0;
	gint nError = 0;

	psDialog = gtk_message_dialog_new( NULL, GTK_DIALOG_MODAL,
		GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, _( "Do you want to delete the list on the fritzbox?" ) );
	gtk_window_set_title( GTK_WINDOW( psDialog ), _( "Question" ) );
	
	nResult = gtk_dialog_run( GTK_DIALOG( psDialog ) );
	gtk_widget_destroy( psDialog );

	switch ( nResult ) {
		case GTK_RESPONSE_YES:
			break;
		default:
			return;
	}

	nError = routerCallListClear( getActiveProfile() );
	if ( nError != 0 ) {
		if ( nError == 33485 ) {
			ShowError( FALSE, _( "Login blocked" ), _( "ffgtk has blocked login until valid password is given" ) );
		} else if ( nError < 0 ) {
			RouterLoginError( FALSE, -nError );
		} else {
			ShowError( FALSE, _( "Network error" ), curl_easy_strerror( nError ) );
		}
	}
}

/**
 * \brief Sort tree model by number, compare a and b
 * \param psModel tree model
 * \param psA first entry tree iterator
 * \param psB second entry tree iterator
 * \param pData UNUSED
 * \return error code
 */
static gint sortByNumber( GtkTreeModel *psModel, GtkTreeIter *psA, GtkTreeIter *psB, gpointer pData ) {
	gint nRet = 0;
	gint nSortCol = GPOINTER_TO_INT( pData );

	switch ( nSortCol ) {
		case COL_CM_TYPE: {
			GdkPixbuf *psPixA;
			GdkPixbuf *psPixB;
		
			gtk_tree_model_get( psModel, psA, nSortCol, &psPixA, -1 );
			gtk_tree_model_get( psModel, psB, nSortCol, &psPixB, -1 );

			if ( psPixA == psPixB ) {
				nRet = 0;
			} else {
				const gchar *pnOptionA = gdk_pixbuf_get_option( psPixA, "title" );
				const gchar *pnOptionB = gdk_pixbuf_get_option( psPixB, "title" );

				if ( pnOptionA == NULL || pnOptionB == NULL ) {
					if ( pnOptionA == NULL && pnOptionB == NULL ) {
						nRet = 0;
					} else {
						nRet = ( pnOptionA == NULL ) ? -1 : 1;
					}
				} else {
					nRet = strcmp( pnOptionA, pnOptionB );
				}
			}
			break;
		}
		case COL_CM_DATETIME: {
			gchar *pnNumberA;
			gchar *pnNumberB;

			gtk_tree_model_get( psModel, psA, nSortCol, &pnNumberA, -1 );
			gtk_tree_model_get( psModel, psB, nSortCol, &pnNumberB, -1 );
			if ( pnNumberA == NULL || pnNumberB == NULL ) {
				if ( pnNumberA == NULL && pnNumberB == NULL ) {
					nRet = 0;
				} else {
					nRet = ( pnNumberA == NULL ) ? -1 : 1;
				}
			} else {
				char *pnPartA1, *pnPartA2, *pnPartA3, *pnPartA4;
				char *pnPartB1, *pnPartB2, *pnPartB3, *pnPartB4;
				int nMonthStart1, nMonthStart2;
				int nYearStart1, nYearStart2;
				int nTimeStart1, nTimeStart2;

				nMonthStart1 = findString( pnNumberA, 0, "." );
				nYearStart1 = findString( pnNumberA, nMonthStart1 + 1, "." );
				nTimeStart1 = findString( pnNumberA, nYearStart1 + 1, " " );
				pnPartA1 = getSubString( pnNumberA, 0, nMonthStart1 );
				pnPartA2 = getSubString( pnNumberA, nMonthStart1 + 1, nYearStart1 - nMonthStart1 - 1 );
				pnPartA3 = getSubString( pnNumberA, nYearStart1 + 1, nTimeStart1 - nYearStart1 - 1 );
				pnPartA4 = getSubString( pnNumberA, nTimeStart1 + 1, strlen( pnNumberA ) - nTimeStart1 - 1 );

				nMonthStart2 = findString( pnNumberB, 0, "." );
				nYearStart2 = findString( pnNumberB, nMonthStart2 + 1, "." );
				nTimeStart2 = findString( pnNumberB, nYearStart2 + 1, " " );
				pnPartB1 = getSubString( pnNumberB, 0, nMonthStart2 );
				pnPartB2 = getSubString( pnNumberB, nMonthStart2 + 1, nYearStart2 - nMonthStart2 - 1 );
				pnPartB3 = getSubString( pnNumberB, nYearStart2 + 1, nTimeStart2 - nYearStart2 - 1 );
				pnPartB4 = getSubString( pnNumberB, nTimeStart2 + 1, strlen( pnNumberB ) - nTimeStart2 - 1 );

				nRet = strcmp( pnPartA3, pnPartB3 );
				if ( nRet == 0 ) {
					nRet = strcmp( pnPartA2, pnPartB2 );
					if ( nRet == 0 ) {
						nRet = strcmp( pnPartA1, pnPartB1 );
						if ( nRet == 0 ) {
							nRet = g_utf8_collate( pnPartA4, pnPartB4 );
						}
					}
				}

				g_free( pnPartA1 );
				g_free( pnPartA2 );
				g_free( pnPartA3 );
				g_free( pnPartA4 );
				g_free( pnPartB1 );
				g_free( pnPartB2 );
				g_free( pnPartB3 );
				g_free( pnPartB4 );
			}
			g_free( pnNumberA );
			g_free( pnNumberB );
			break;
		}
		case COL_CM_NAME:
		case COL_CM_COMPANY:
		case COL_CM_CITY:
		case COL_CM_NUMBER:
		case COL_CM_LOCALNAME:
		case COL_CM_LOCALNUMBER:
		case COL_CM_DURATION: {
			gchar *pnNumberA;
			gchar *pnNumberB;

			gtk_tree_model_get( psModel, psA, nSortCol, &pnNumberA, -1 );
			gtk_tree_model_get( psModel, psB, nSortCol, &pnNumberB, -1 );
			if ( pnNumberA == NULL || pnNumberB == NULL ) {
				if ( pnNumberA == NULL && pnNumberB == NULL ) {
					nRet = 0;
				} else {
					nRet = ( pnNumberA == NULL ) ? -1 : 1;
				}
			} else {
				//nRet = g_utf8_collate( pnNumberA, pnNumberB );
				nRet = strcmp( pnNumberA, pnNumberB );
			}
			g_free( pnNumberA );
			g_free( pnNumberB );
			break;
		}
	}


	return nRet;
}

/**
 * \brief Show dialog while calling number and give option to hangup
 * \param pnNumber number to dial
 */
static void callNumber( const gchar *pnNumber ) {
	struct sPerson *psPerson = findPersonByNumber( pnNumber );

	dialNumberDialog( psPerson, pnNumber );
}

/**
 * \brief Update caller list (treeview and disk file)
 * \param pnNumber number we want to add the name to
 * \param pnName name
 */
void updateCallerList( const gchar *pnNumber, gchar *pnName ) {
	GtkTreeIter sIter;
	gboolean bValid = FALSE;

	if ( pnNumber != NULL ) {
		/* Update current callmonitor tree view */
		bValid = gtk_tree_model_get_iter_first( psListStore, &sIter );

		while ( bValid ) {
			gchar *pnCallNumber;
			gtk_tree_model_get( psListStore, &sIter, COL_CM_NUMBER, &pnCallNumber, -1 );

			if ( strcmp( pnCallNumber, pnNumber ) == 0 ) {
				gtk_list_store_set( GTK_LIST_STORE( psListStore ), &sIter, COL_CM_NAME, pnName, -1 );
			}
			g_free( pnCallNumber );
			bValid = gtk_tree_model_iter_next( psListStore, &sIter );
		}

		/* Update internal structure */
		GList *psList = NULL;
		struct sCaller *psCaller = NULL;

		for ( psList = psCallerList; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
			psCaller = psList -> data;

			if ( psCaller -> pnNumber == NULL || strlen( psCaller -> pnNumber ) <= 0 ) {
				continue;
			}

			gchar *pnFullNumber = fullNumber( psCaller -> pnNumber, FALSE );

			if ( strcmp( pnFullNumber, pnNumber ) == 0 ) {
				if ( psCaller -> pnName != NULL ) {
					g_free( psCaller -> pnName );
				}
				psCaller -> pnName = g_strdup( pnName );
			}
			g_free( pnFullNumber );
		}
	}

	/* Save to disk */
	saveCallerList();
}

/**
 * \brief Add entry to address book callback
 * \param psListStore list store widget
 * \param pnNumber telephone number
 */
static void callMonitorAddEntry( GtkListStore *psListStore, const gchar *pnNumber ) {
	GtkBuilder *psBuilder = NULL;
	GError *psError = NULL;
	GtkWidget *psDialog = NULL;
	GtkWidget *psExistingRadio = NULL;
	gchar *pnUiFile;
	gint nResult;
	gboolean bExisting = FALSE;

	psBuilder = gtk_builder_new();
	pnUiFile = getUiFile( "kind.ui" );
	if ( gtk_builder_add_from_file( psBuilder, pnUiFile, &psError ) == 0 ) {
	    Debug( KERN_WARNING, "Error: %s\n", psError -> message );
	    g_error_free( psError );
		g_free( pnUiFile );
	    return;
	}
	g_free( pnUiFile );

	psDialog = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psKindDialog" ) );
	psExistingRadio = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psExistingRadio" ) );

	gtk_builder_connect_signals( psBuilder, NULL );

	g_object_unref( G_OBJECT( psBuilder ) );

	nResult = gtk_dialog_run( GTK_DIALOG( psDialog ) );
	bExisting = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psExistingRadio ) );

	gtk_widget_destroy( GTK_WIDGET( psDialog ) );

	if ( nResult == 2 ) {
		gtk_clipboard_set_text( gtk_clipboard_get( GDK_NONE ), pnNumber, -1 );

		if ( bExisting == TRUE ) {
			AddressBook( 2 );
		} else {
			AddAddressDialog( NULL, pnNumber, 0 );
			AddressBook( 0 );
			SaveBook();
		}
	}
}

/**
 * \brief Callmonitor lookup window response - callback
 * \param psWidget window
 * \param nResponse response id
 * \param pUserData telephone number
 */
static void callMonitorLookupAdd( GtkWidget *psWidget, gint nResponse, gpointer pUserData ) {
	GtkWidget *psNumber = g_object_get_data( G_OBJECT( psWidget ), "number" );
	GtkWidget *psFirstName = g_object_get_data( G_OBJECT( psWidget ), "firstname" );
	GtkWidget *psLastName = g_object_get_data( G_OBJECT( psWidget ), "lastname" );
	GtkWidget *psStreet = g_object_get_data( G_OBJECT( psWidget ), "street" );
	GtkWidget *psZipCode = g_object_get_data( G_OBJECT( psWidget ), "zipcode" );
	GtkWidget *psCity = g_object_get_data( G_OBJECT( psWidget ), "city" );
	const gchar *pnNumber = gtk_entry_get_text( GTK_ENTRY( psNumber ) );
	const gchar *pnFirstName = gtk_entry_get_text( GTK_ENTRY( psFirstName ) );
	const gchar *pnLastName = gtk_entry_get_text( GTK_ENTRY( psLastName ) );
	const gchar *pnStreet = gtk_entry_get_text( GTK_ENTRY( psStreet ) );
	const gchar *pnZipCode = gtk_entry_get_text( GTK_ENTRY( psZipCode ) );
	const gchar *pnCity = gtk_entry_get_text( GTK_ENTRY( psCity ) );
	char *pnFullName = NULL;

	if ( nResponse == 1 ) {
		/* Add pressed */
		GHashTable *psTable = g_hash_table_new( NULL, NULL );
		struct sPerson *psPerson = NULL;
	
		pnFullName = g_strdup_printf("%s %s", pnFirstName, pnLastName);

		AddInfo( psTable, PERSON_ID, pnFullName );

		if ( pnNumber != NULL && strlen( pnNumber ) > 0 ) {
			AddInfo( psTable, PERSON_PRIVATE_PHONE, pnNumber );
		}

		if ( pnStreet != NULL && strlen( pnStreet ) > 0 ) {
			AddInfo( psTable, PERSON_PRIVATE_STREET, pnStreet );
		}

		if ( pnZipCode != NULL && strlen( pnZipCode ) > 0 ) {
			AddInfo( psTable, PERSON_PRIVATE_ZIPCODE, pnZipCode );
		}

		if ( pnCity != NULL && strlen( pnCity ) > 0 ) {
			AddInfo( psTable, PERSON_PRIVATE_CITY, pnCity );
		}

		if ( pnLastName != NULL && strlen( pnLastName ) > 0 ) {
			AddInfo( psTable, PERSON_LAST_NAME, pnLastName );
		}
		if ( pnFirstName != NULL && strlen( pnFirstName ) > 0 ) {
			AddInfo( psTable, PERSON_FIRST_NAME, pnFirstName );
		}

		AddInfo( psTable, PERSON_DISPLAY_NAME, pnFullName );

		psPerson = AddPerson( psTable, TRUE );
		if ( psPerson != NULL ) {
			SaveBook();
			GtkWidget *psInfoDialog = gtk_message_dialog_new( GTK_WINDOW( psWidget ), GTK_DIALOG_DESTROY_WITH_PARENT, 
						GTK_MESSAGE_INFO, GTK_BUTTONS_OK, _( "Entry '%s' successfully added" ), pnFullName );
			gtk_dialog_run( GTK_DIALOG( psInfoDialog ) );
			gtk_widget_destroy( psInfoDialog );
		}

		updateCallerList( ( gchar * ) pnNumber, ( gchar * ) pnLastName );

		g_hash_table_destroy( psTable );

		if ( pnFullName != NULL ) {
			g_free( pnFullName );
		}
	}

	gtk_widget_destroy( GTK_WIDGET( psWidget ) );
}

/**
 * \brief Call-Monitor find entry callback
 * \param psWidget button widget
 * \param pUserData callmonitor treeview
 */
static void callMonitorFind( GtkWidget *psWidget, gpointer pUserData ) {
	GtkWidget *psDialog;
	GtkWidget *psTable;
	GtkWidget *psFirstName;
	GtkWidget *psFirstNameEntry;
	GtkWidget *psLastName;
	GtkWidget *psLastNameEntry;
	GtkWidget *psStreet;
	GtkWidget *psStreetEntry;
	GtkWidget *psZipCode;
	GtkWidget *psZipCodeEntry;
	GtkWidget *psCity;
	GtkWidget *psCityEntry;
	GtkWidget *psNumber;
	GtkWidget *psNumberEntry;
	GtkWidget *psBox;
	gchar *pnFirstName;
	gchar *pnLastName;
	gchar *pnStreet;
	gchar *pnZipCode;
	gchar *pnCity;
	gint nWidth, nHeight;
	gint nError = -1;
	GValue sNumber = { 0 };
	gchar *pnNumber = NULL;	
	GtkTreeIter sSelectedIter;
	GtkTreeModel *psModel;
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( GTK_TREE_VIEW( pUserData ) );
	GList *psPathList;

	psPathList = gtk_tree_selection_get_selected_rows( psSelection, &psModel );
	if ( psPathList == NULL ) {
		return;
	}

	if ( g_list_length( psPathList ) == 1 ) {
		GtkTreePath *psPath = psPathList -> data;

		if ( gtk_tree_model_get_iter( psModel, &sSelectedIter, psPath ) ) {
			gtk_tree_model_get_value( psModel, &sSelectedIter, COL_CM_NUMBER, &sNumber );
			pnNumber = g_strdup( g_value_get_string( &sNumber ) );
			g_value_unset( &sNumber );
		} else {
			return;
		}
	} else {
		return;
	}

	nError = ReverseLookup( pnNumber, &pnFirstName, &pnLastName, &pnStreet, &pnZipCode, &pnCity );

	if ( nError != 0 ) {
		ShowError( FALSE, _( "Reverse lookup failed" ), _( "Entry not found" ) );//curl_easy_strerror( nError ) );
		g_free( pnNumber );
		return;
	}

	if ( ( pnFirstName != NULL ) || ( pnLastName != NULL ) ) {
		psDialog = gtk_dialog_new_with_buttons( _( "Reverse lookup report:" ), NULL, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_STOCK_ADD, 1, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL );
	} else {
		psDialog = gtk_dialog_new_with_buttons( _( "Reverse lookup report:" ), NULL, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL );
	}

	gtk_window_set_title( GTK_WINDOW( psDialog ), _( "Information" ) );
	g_signal_connect( psDialog, "response", G_CALLBACK( callMonitorLookupAdd ), pUserData );

	psTable = gtk_table_new( 5, 2, false );
	gtk_container_set_border_width( GTK_CONTAINER( psTable ), 5 );

	psFirstName = gtk_label_new( _( "First name:" ) );
	gtk_misc_set_alignment( GTK_MISC( psFirstName ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( psFirstName ), 25, 0 );
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psFirstName ), 0, 1, 0, 1, GTK_FILL, 0, 0, 0 );

	psFirstNameEntry = gtk_entry_new();
	if ( pnFirstName != NULL ) {
		gtk_entry_set_text( GTK_ENTRY( psFirstNameEntry ), pnFirstName );
		g_free( pnFirstName );
	} else {
		gtk_widget_set_sensitive( GTK_WIDGET( psFirstNameEntry ), false );
	}
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psFirstNameEntry ), 1, 2, 0, 1, GTK_FILL | GTK_EXPAND, 0, 0, 0 );

	psLastName = gtk_label_new( _( "Last name:" ) );
	gtk_misc_set_alignment( GTK_MISC( psLastName ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( psLastName ), 25, 0 );
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psLastName ), 0, 1, 1, 2, GTK_FILL, 0, 0, 0 );

	psLastNameEntry = gtk_entry_new();
	if ( pnLastName != NULL ) {
		gtk_entry_set_text( GTK_ENTRY( psLastNameEntry ), pnLastName );
		g_free( pnLastName );
	} else {
		gtk_widget_set_sensitive( GTK_WIDGET( psLastNameEntry ), false );
	}
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psLastNameEntry ), 1, 2, 1, 2, GTK_FILL | GTK_EXPAND, 0, 0, 0 );

	psStreet = gtk_label_new( _( "Street:" ) );
	gtk_misc_set_alignment( GTK_MISC( psStreet ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( psStreet ), 25, 0 );
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psStreet ), 0, 1, 2, 3, GTK_FILL, 0, 0, 0 );

	psStreetEntry = gtk_entry_new();
	if ( pnStreet != NULL ) {
		gtk_entry_set_text( GTK_ENTRY( psStreetEntry ), pnStreet );
		g_free( pnStreet );
	} else {
		gtk_widget_set_sensitive( GTK_WIDGET( psStreetEntry ), false );
	}
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psStreetEntry ), 1, 2, 2, 3, GTK_FILL | GTK_EXPAND, 0, 0, 0 );

	psZipCode = gtk_label_new( _( "Zip code:" ) );
	gtk_misc_set_alignment( GTK_MISC( psZipCode ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( psZipCode ), 25, 0 );
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psZipCode ), 0, 1, 3, 4, GTK_FILL, 0, 0, 0 );

	psZipCodeEntry = gtk_entry_new();
	if ( pnZipCode != NULL ) {
		gtk_entry_set_text( GTK_ENTRY( psZipCodeEntry ), pnZipCode );
		g_free( pnZipCode );
	} else {
		gtk_widget_set_sensitive( GTK_WIDGET( psZipCodeEntry ), false );
	}
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psZipCodeEntry ), 1, 2, 3, 4, GTK_FILL | GTK_EXPAND, 0, 0, 0 );

	psCity = gtk_label_new( _( "City:" ) );
	gtk_misc_set_alignment( GTK_MISC( psCity ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( psCity ), 25, 0 );
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psCity ), 0, 1, 4, 5, GTK_FILL, 0, 0, 0 );

	psCityEntry = gtk_entry_new();
	if ( pnCity != NULL ) {
		gtk_entry_set_text( GTK_ENTRY( psCityEntry ), pnCity );
		g_free( pnCity );
	} else {
		gtk_widget_set_sensitive( GTK_WIDGET( psCityEntry ), false );
	}
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psCityEntry ), 1, 2, 4, 5, GTK_FILL | GTK_EXPAND, 0, 0, 0 );

	psNumber = gtk_label_new( _( "Number:" ) );
	gtk_misc_set_alignment( GTK_MISC( psNumber ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( psNumber ), 25, 0 );
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psNumber ), 0, 1, 5, 6, GTK_FILL, 0, 0, 0 );

	psNumberEntry = gtk_entry_new();
	gtk_entry_set_text( GTK_ENTRY( psNumberEntry ), pnNumber );
	gtk_table_attach( GTK_TABLE( psTable ), GTK_WIDGET( psNumberEntry ), 1, 2, 5, 6, GTK_FILL | GTK_EXPAND, 0, 0, 0 );
	g_free( pnNumber );

	psBox = gtk_dialog_get_content_area( GTK_DIALOG( psDialog ) );
	if ( psBox != NULL ) {
		gtk_box_pack_end( GTK_BOX( psBox ), psTable, TRUE, FALSE, 5 );
	}

	g_object_set_data( G_OBJECT( psDialog ), "number", psNumberEntry );
	g_object_set_data( G_OBJECT( psDialog ), "firstname", psFirstNameEntry );
	g_object_set_data( G_OBJECT( psDialog ), "lastname", psLastNameEntry );
	g_object_set_data( G_OBJECT( psDialog ), "street", psStreetEntry );
	g_object_set_data( G_OBJECT( psDialog ), "zipcode", psZipCodeEntry );
	g_object_set_data( G_OBJECT( psDialog ), "city", psCityEntry );

	callMonitorGetSize( getActiveProfile(), &nWidth, &nHeight );

	gtk_window_set_default_size( GTK_WINDOW( psDialog ), nWidth / 2, -1 );
	gtk_widget_show_all( GTK_WIDGET( psDialog ) );
}

/**
 * \brief Column clicked, set search column!
 * \param psColumn clicked column
 * \param pData treeview
 */
static void columnClicked( GtkTreeViewColumn *psColumn, gpointer pData ) {
	GtkTreeView *psView = GTK_TREE_VIEW( pData );
	gint nId = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( psColumn ), "model-column" ) );

	gtk_tree_view_set_search_column( psView, nId );
}

/**
 * \brief Call Monitor configure callback
 * \param psWidget window widget
 * \param pUserData userdata pointer UNUSED
 * \return TRUE
 */
static gboolean callMonitorConfigure( GtkWidget *psWidget, gpointer pUserData ) {
	gtk_window_get_size( GTK_WINDOW( psWidget ), &nCallMonitorWidth, &nCallMonitorHeight );

	return FALSE;
}

/**
 * \brief Response on callmonitor close, set tray open flag to false and destroy widget
 * \param psWidget callmonitor
 * \param nResult result id
 * \param pUserData UNUSED
 * \return handle event flag
 */
static gboolean callMonitorResponse( GtkWidget *psWidget, gint nResult, gpointer pUserData ) {
	setTrayOpen( false );
	gtk_widget_hide( GTK_WIDGET( psWidget ) );
	callMonitorSetSize( getActiveProfile(), nCallMonitorWidth, nCallMonitorHeight );
	SavePreferences( getActiveProfile() );

	return TRUE;
}

/**
 * \brief Call-Monitor add entry callback
 * \param psWidget button widget
 * \param pUserData call monitor tree view
 */
static void callMonitorAdd( GtkWidget *psWidget, gpointer pUserData ) {
	GtkTreeIter sSelectedIter;
	GtkTreeModel *psModel;
	GtkTreeView *psTreeView = GTK_TREE_VIEW( pUserData );
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( psTreeView );
	GValue sNumber = { 0 };
	GList *psPathList = NULL;

	psPathList = gtk_tree_selection_get_selected_rows( psSelection, &psModel );
	if ( psPathList == NULL ) {
		return;
	}

	if ( g_list_length( psPathList ) == 1 ) {
		GtkTreePath *psPath = psPathList -> data;

		if ( gtk_tree_model_get_iter( psModel, &sSelectedIter, psPath ) ) {
			gtk_tree_model_get_value( psModel, &sSelectedIter, COL_CM_NUMBER, &sNumber );

			callMonitorAddEntry( GTK_LIST_STORE( gtk_tree_view_get_model( psTreeView ) ), g_value_get_string( &sNumber ) );

			g_value_unset( &sNumber );
		}
	}

	g_list_foreach( psPathList, ( GFunc ) gtk_tree_path_free, NULL );
	g_list_free( psPathList );
}

/**
 * \brief Remove one single call list entry
 * \param psWidget widget
 * \param pnType call type
 * \param pnDate date/timestamp
 */
void removeCallListEntry( GtkWidget *psWidget, const gchar *pnType, const gchar *pnDate ) {
	GList *psList;

	for ( psList = psCallerList; psList != NULL; psList = psList -> next ) {
		struct sCaller *psCaller = psList -> data;

		if ( psCaller != NULL ) {
			if ( strcmp( psCaller -> pnDateTime, pnDate ) == 0 ) {
				psCallerList = g_list_remove( psCallerList, psCaller );
				saveCallerList();
				return;
			}
		}
	}
}

/**
 * \brief Call-Monitor delete entry callback
 * \param psWidget button widget
 * \param pUserData call monitor tree view
 */
static void callMonitorDelete( GtkWidget *psWidget, gpointer pUserData ) {
	GtkTreeIter sSelectedIter;
	GtkTreeModel *psModel;
	GtkTreeView *psTreeView = GTK_TREE_VIEW( pUserData );
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( psTreeView );
	GdkPixbuf *psPixBuf;
	const gchar *pnName = NULL;
	const gchar *pnDate = NULL;
	GList *psList = NULL;
	GList *psPathList = NULL;
	GtkWidget *psDialog = NULL;
	gint nResult = 0;
	GtkListStore *psListStore = pUserData;

	psPathList = gtk_tree_selection_get_selected_rows( psSelection, &psModel );
	if ( psPathList == NULL ) {
		return;
	}

	psDialog = gtk_message_dialog_new( NULL, GTK_DIALOG_MODAL,
			GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, _( "Do you want to delete the selected entry?" ) );
	gtk_window_set_title( GTK_WINDOW( psDialog ), _( "Question" ) );
	
	nResult = gtk_dialog_run( GTK_DIALOG( psDialog ) );
	gtk_widget_destroy( psDialog );

	if ( nResult == GTK_RESPONSE_YES ) {
		for ( psList = psPathList; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
			GtkTreePath *psPath = psList -> data;

			if ( gtk_tree_model_get_iter( psModel, &sSelectedIter, psPath ) ) {
				gtk_tree_model_get( psModel, &sSelectedIter, COL_CM_TYPE, &psPixBuf, -1 );
				if ( psPixBuf != NULL ) {
					pnName = gdk_pixbuf_get_option( psPixBuf, "title" );
					Debug( KERN_DEBUG, "pnName: %s\n", pnName );

					if ( strcmp( pnName, "faxbox" ) == 0 ) {
						faxBoxDelete( psModel, psPath );
					} else if ( strcmp( pnName, "voicebox" ) == 0 ) {
						voiceBoxDelete( psModel, psPath );
					} else {
						gtk_tree_model_get( psModel, &sSelectedIter, COL_CM_DATETIME, &pnDate, -1 );
						removeCallListEntry( psWidget, pnName, pnDate );
					}
				}
			}
		}

		psListStore = GTK_LIST_STORE( gtk_tree_view_get_model(pUserData) );
		/* Clear list store */
		gtk_list_store_clear( psListStore );
		/* Clear internal caller structure */
		freeCallers();
		/* Read caller list */
		loadCallerList();
		/* Add callers to list store */
		displayCallerList( psListStore, psCurrentFilter );
	}

	g_list_foreach( psPathList, ( GFunc ) gtk_tree_path_free, NULL );
	g_list_free( psPathList );
}

/**
 * \brief CallMonitor selection pressed, switch to selected type
 * \param psTreeView selection treeview
 * \param psEvent button event occurred
 * \param pUserData UNUSED
 */
static gboolean selectionPressed( GtkTreeView *psTreeView, GdkEventButton *psEvent, gpointer pUserData ) {
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( psTreeView );
	GtkTreeModel *psModel;
	GtkTreeIter sIter;

	if ( ( psEvent -> type != GDK_BUTTON_PRESS ) || ( psEvent -> button != 1 ) ) {
		return FALSE;
	}

	if ( gtk_tree_selection_count_selected_rows( psSelection ) <= 1 ) {
		GtkTreePath *psPath;
		gchar *pnName;

		/* Get tree path for row that was clicked */
		if ( gtk_tree_view_get_path_at_pos( GTK_TREE_VIEW( psTreeView ), (gint) psEvent -> x, (gint) psEvent -> y, &psPath, NULL, NULL, NULL ) ) {
			gtk_tree_selection_unselect_all( psSelection );
			gtk_tree_selection_select_path( psSelection, psPath );

			psModel = gtk_tree_view_get_model( psTreeView );
			if ( gtk_tree_model_get_iter( psModel, &sIter, psPath ) ) {
				gtk_tree_model_get( psModel, &sIter, 0, &pnName, -1 );
				if ( pnName != NULL ) {
					gtk_list_store_clear( pUserData );

					psCurrentFilter = findFilter( pnName );
					displayCallerList( pUserData, psCurrentFilter );

					gtk_tree_path_free( psPath );
					return TRUE;
				}
			}
			gtk_tree_path_free( psPath );
		}
	}

	return FALSE;
}

/**
 * \brief CallMonitor pressed, set button states
 * \param psTreeView selection treeview
 * \param psEvent button event occured
 * \param pUserData UNUSED
 */
static gboolean callMonitorPressed( GtkTreeView *psTreeView, GdkEventButton *psEvent, gpointer pUserData ) {
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( psTreeView );
	GtkTreeModel *psModel;
	GtkTreeIter sSelectedIter;
	GdkPixbuf *psPixBuf = NULL;
	GList *psPathList = NULL;
	gboolean bHandled = FALSE;

	if ( psEvent -> window != gtk_tree_view_get_bin_window( psTreeView ) ) {
		Debug( KERN_INFO, "Exiting...\n" );
		return FALSE;
	}

	if ( ( psEvent -> type != GDK_2BUTTON_PRESS ) || ( psEvent -> button != 1 ) ) {
		Debug( KERN_INFO, "Type: %d, Button: %d\n", psEvent -> type, psEvent -> button );
		return FALSE;
	}

	psPathList = gtk_tree_selection_get_selected_rows( psSelection, &psModel );
	if ( psPathList == NULL ) {
		return FALSE;
	}

	if ( g_list_length( psPathList ) == 1 ) {
		GtkTreePath *psPath = psPathList -> data;
		const gchar *pnName;

		if ( gtk_tree_model_get_iter( psModel, &sSelectedIter, psPath ) ) {
			gtk_tree_model_get( psModel, &sSelectedIter, COL_CM_TYPE, &psPixBuf, -1 );

			if ( psPixBuf != NULL ) {
				pnName = gdk_pixbuf_get_option( psPixBuf, "title" );
				if ( strcmp( pnName, "faxbox" ) == 0 ) {
					faxBoxOpen( psModel, psPath );
				} else if ( strcmp( pnName, "voicebox" ) == 0 ) {
					voiceBoxPlay( psModel, psPath );
				} else {
					GValue sNumber = { 0 };

					gtk_tree_model_get_value( psModel, &sSelectedIter, COL_CM_NUMBER, &sNumber );

					callNumber( g_value_get_string( &sNumber ) );

					g_value_unset( &sNumber );
				}
				bHandled = TRUE;
			}
		}
	}

	g_list_foreach( psPathList, ( GFunc ) gtk_tree_path_free, NULL );
	g_list_free( psPathList );

	return bHandled;
}

/**
 * \brief Handle paned press event - store position
 * \param psWidget paned widget
 * \param psEvent event structure
 * \param pData user data UNUSED
 * \return FALSE
 */
static gboolean panedPressed( GtkWidget *psWidget, GdkEventButton *psEvent, gpointer pData ) {
	nPressPosition = gtk_paned_get_position( GTK_PANED( psWidget ) );
	return FALSE;
}

/**
 * \brief Handle paned release event - hide/show pane
 * \param psWidget paned widget
 * \param psEvent event structure
 * \param pData user data UNUSED
 * \return FALSE
 */
static gboolean panedReleased( GtkWidget *psWidget, GdkEventButton *psEvent, gpointer pData ) {
	static int nPosition = 0;

	if ( nPressPosition == -1 ) {
		return FALSE;
	}

	if ( abs( nPressPosition - gtk_paned_get_position( GTK_PANED( psWidget ) ) ) < 4 ) {
		if ( gtk_paned_get_position( GTK_PANED( psWidget ) ) > 0 ) {
			nPosition = gtk_paned_get_position( GTK_PANED( psWidget ) );
			gtk_paned_set_position( GTK_PANED( psWidget ), 0 );
		} else {
			if ( nPosition == 0 ) {
				nPosition = 180;
			}
			gtk_paned_set_position( GTK_PANED( psWidget ), nPosition );
		}
	}

	callMonitorSetPanedPosition( getActiveProfile(), gtk_paned_get_position( GTK_PANED( psWidget ) ) );
	saveProfiles();
	nPressPosition = -1;

	return FALSE;
}

/**
 * \brief Lookup combo box changed, update internal structures
 * \param psWidget lookup combo box widget
 * \param pUserData UNUSED
 */
static void callMonitorLookupChanged( GtkWidget *psWidget, gpointer pUserData ) {
	pluginsSetLookup( getActiveProfile(), gtk_combo_box_get_active_text( GTK_COMBO_BOX( psWidget ) ) );
	setDefaultPlugin( getActiveProfile(), PLUGIN_TYPE_LOOKUP, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psWidget ) ) );
}

/**
 * \brief Set cell foreground color depending on entry text
 * \param psCol treeview column
 * \param psRenderer cell renderer
 * \param psModel tree model
 * \param psIter tree iterator
 * \param pUserData user data UNUSED
 */
static void nameCellData( GtkTreeViewColumn *psCol, GtkCellRenderer *psRenderer, GtkTreeModel *psModel, GtkTreeIter *psIter, gpointer pUserData ) {
	const gchar *pnName;

	gtk_tree_model_get( psModel, psIter, COL_CM_NAME, &pnName, -1 );

	if ( pnName != NULL && strlen( pnName ) > 0 && pnName[ 0 ] == '(' ) {
		g_object_set( psRenderer, "foreground", "DarkGrey", "foreground-set", TRUE, NULL );
	} else {
		g_object_set( psRenderer, "foreground-set", FALSE, NULL );
	}
}

/**
 * \brief Open callmonitor and return widget
 * \return CallMonitor widget
 */
GtkWidget *CallMonitor( void ) {
	GtkBuilder *psBuilder = NULL;
	GError *psError = NULL;
	GtkWidget *psSearchEntry = NULL;
	GtkWidget *psDelButton = NULL;
	GtkWidget *psAddButton = NULL;
	GtkWidget *psEditButton = NULL;
	GtkWidget *psPaned = NULL;
	GtkCellRenderer *psRenderer = NULL;
	GtkTreeViewColumn *psColumn = NULL;
	GtkTreeSortable *psSortable = NULL;
	GtkToolItem *psToolItem = NULL;
	static GtkToolItem *psAddToolItem = NULL;
	static GtkToolItem *psAllProfilesItem = NULL;
	static GtkToolItem *psNextProfileItem = NULL;
	GtkTreeIter sIter;
	GtkTreeModel *psModel = NULL;
	GtkListStore *psLookupStore = NULL;
	GtkTreeSelection *psSelection = NULL;
	GList *psList = NULL;
	struct sLookup *psLookup = NULL;
	struct sAddressBook *psBook = NULL;
	gchar *pnUiFile = NULL;
	gchar *pnDummy = NULL;
	gint nCount = 0;
	GList *psProfiles = getProfiles();

	psCallMonitorProfile = getActiveProfile();
	if ( psCallMonitorProfile == NULL ) {
		Debug( KERN_WARNING, "No profile active, abort!\n" );
		ShowError( FALSE, _( "Abort" ), _( "No profile is active" ) );
		return NULL;
	}

	if ( psCallMonitor != NULL ) {
		/* Set window title */
		pnDummy = getProfileCaption();
		gtk_window_set_title( GTK_WINDOW( psCallMonitor ), pnDummy );
		g_free( pnDummy );

		/* Clear caller list and reload data */
		gtk_list_store_clear( psMonitorStore );
		displayCallerList( psMonitorStore, psCurrentFilter );

		/* Update filter */
		gtk_list_store_clear( psFilterStore );
		updateFilter( psFilterStore, psSelectionTreeView );
		if ( gtk_tree_model_get_iter_first( GTK_TREE_MODEL( psListStore ), &sIter ) == TRUE ) {
			gtk_tree_selection_select_iter( gtk_tree_view_get_selection( GTK_TREE_VIEW( psCallMonitorTreeView ) ), &sIter );
		}

		switch ( miscGetToolbarStyle( getActiveProfile() ) ) {
			case 0:
				/* default */
				gtk_toolbar_unset_style( GTK_TOOLBAR( psToolbar ) );
				break;
			case 1:
				/* text below icons */
				gtk_toolbar_set_style( GTK_TOOLBAR( psToolbar ), GTK_TOOLBAR_BOTH );
				break;
			case 2:
				/* text beside icons */
				gtk_toolbar_set_style( GTK_TOOLBAR( psToolbar ), GTK_TOOLBAR_BOTH_HORIZ );
				break;
			case 3:
				/* icons only */
				gtk_toolbar_set_style( GTK_TOOLBAR( psToolbar ), GTK_TOOLBAR_ICONS );
				break;
			case 4:
				/* text only */
				gtk_toolbar_set_style( GTK_TOOLBAR( psToolbar ), GTK_TOOLBAR_TEXT );
				break;
			default:
				Debug( KERN_WARNING, "Unknown toolbar style, setting default\n" );
				gtk_toolbar_set_style( GTK_TOOLBAR( psToolbar ), GTK_TOOLBAR_TEXT );
				break;
		}

		gtk_tree_view_column_set_visible( psTypeColumn, callMonitorGetType( getActiveProfile() ) );
		gtk_tree_view_column_set_visible( psDateTimeColumn, callMonitorGetDateTime( getActiveProfile() ) );
		gtk_tree_view_column_set_visible( psNameColumn, callMonitorGetName( getActiveProfile() ) );
		gtk_tree_view_column_set_visible( psCompanyColumn, callMonitorGetCompany( getActiveProfile() ) );
		gtk_tree_view_column_set_visible( psCityColumn, callMonitorGetCity( getActiveProfile() ) );
		gtk_tree_view_column_set_visible( psNumberColumn, callMonitorGetNumber( getActiveProfile() ) );
		gtk_tree_view_column_set_visible( psLocalNameColumn, callMonitorGetLocalName( getActiveProfile() ) );
		gtk_tree_view_column_set_visible( psLocalNumberColumn, callMonitorGetLocalNumber( getActiveProfile() ) );
		gtk_tree_view_column_set_visible( psDurationColumn, callMonitorGetDuration( getActiveProfile() ) );

		psBook = getDefaultBookPlugin( getActiveProfile() );
		if ( psBook == NULL || psBook -> SaveBook == NULL ) {
			gtk_widget_set_sensitive( GTK_WIDGET( psAddToolItem ), FALSE );
		} else {
			gtk_widget_set_sensitive( GTK_WIDGET( psAddToolItem ), TRUE );
		}

		if ( psProfiles == NULL || g_list_length( psProfiles ) < 2 ) {
			gtk_widget_set_sensitive( GTK_WIDGET( psAllProfilesItem ), FALSE );
			gtk_widget_set_sensitive( GTK_WIDGET( psNextProfileItem ), FALSE );
		} else {
			gtk_widget_set_sensitive( GTK_WIDGET( psAllProfilesItem ), TRUE );
			gtk_widget_set_sensitive( GTK_WIDGET( psNextProfileItem ), TRUE );
		}

		gtk_widget_show( GTK_WIDGET( psCallMonitor ) );

		return psCallMonitor;
	}

	LoadFilters();

	psBuilder = gtk_builder_new();
	pnUiFile = getUiFile( "callmonitor.ui" );
	if ( gtk_builder_add_from_file( psBuilder, pnUiFile, &psError ) == 0 ) {
	    Debug( KERN_WARNING, "Error: %s\n", psError -> message );
	    g_error_free( psError );
		g_free( pnUiFile );
	    return NULL;
	}
	g_free( pnUiFile );

	psCurrentFilter = NULL;

	psCallMonitor = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallMonitorWindow" ) );
	psCallMonitorTreeView = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallMonitorTreeView" ) );
	psSearchEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psSearchEntry" ) );
	gtk_tree_view_set_search_entry( GTK_TREE_VIEW( psCallMonitorTreeView ), GTK_ENTRY( psSearchEntry ) );

	psSelectionTreeView = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psSelectionTreeView" ) );
	psFilterStore = gtk_list_store_new( 3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING );
	psListStore = GTK_TREE_MODEL( psFilterStore );
	gtk_tree_view_set_model( GTK_TREE_VIEW( psSelectionTreeView ), GTK_TREE_MODEL( psListStore ) );

	psPaned = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPaned" ) );
	g_signal_connect( G_OBJECT( psPaned ), "button-press-event", G_CALLBACK( panedPressed ), NULL );
	g_signal_connect( G_OBJECT( psPaned ), "button-release-event", G_CALLBACK( panedReleased ), NULL );

	psAddButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psAddButton" ) );
	g_signal_connect( G_OBJECT( psAddButton ), "clicked", G_CALLBACK( filterAdd ), psFilterStore );
	psDelButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDelButton" ) );
	g_signal_connect( G_OBJECT( psDelButton ), "clicked", G_CALLBACK( filterDel ), psFilterStore );
	psEditButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psEditButton" ) );
	g_signal_connect( G_OBJECT( psEditButton ), "clicked", G_CALLBACK( filterEdit ), psFilterStore );

	psRenderer = gtk_cell_renderer_text_new();
	psColumn = gtk_tree_view_column_new_with_attributes( _( "Selection" ), psRenderer, "text", COL_FILTER_TYPE, NULL );
	gtk_tree_view_column_set_sort_column_id( psColumn, COL_FILTER_TYPE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psSelectionTreeView ), psColumn );

	psRenderer = gtk_cell_renderer_text_new();
	psColumn = gtk_tree_view_column_new_with_attributes( _( "Count" ), psRenderer, "text", COL_FILTER_COUNT, NULL );
	gtk_tree_view_column_set_sort_column_id( psColumn, COL_FILTER_COUNT );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psSelectionTreeView ), psColumn );

	psRenderer = gtk_cell_renderer_text_new();
	psColumn = gtk_tree_view_column_new_with_attributes( _( "Duration" ), psRenderer, "text", COL_FILTER_DURATION, NULL );
	gtk_tree_view_column_set_sort_column_id( psColumn, COL_FILTER_DURATION );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psSelectionTreeView ), psColumn );
	
	psMonitorStore = gtk_list_store_new( 9,
		GDK_TYPE_PIXBUF,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
        G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_STRING );
	psListStore = GTK_TREE_MODEL( psMonitorStore );
	psSortable = GTK_TREE_SORTABLE( psListStore );

	gtk_tree_view_set_model( GTK_TREE_VIEW( psCallMonitorTreeView ), GTK_TREE_MODEL( psListStore ) );

	g_signal_connect( G_OBJECT( psSelectionTreeView ), "button-press-event", G_CALLBACK( selectionPressed ), psMonitorStore );

	loadCallerList();
	displayCallerList( psMonitorStore, psCurrentFilter );

	psRenderer = gtk_cell_renderer_pixbuf_new();
	psTypeColumn = gtk_tree_view_column_new_with_attributes( _( "Type" ), psRenderer, "pixbuf", COL_CM_TYPE, NULL );
	gtk_tree_view_column_set_sort_column_id( psTypeColumn, COL_CM_TYPE );
	gtk_tree_view_column_set_reorderable( psTypeColumn, TRUE );
	gtk_tree_view_column_set_resizable( psTypeColumn, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorTreeView ), psTypeColumn );
	gtk_tree_view_column_set_visible( psTypeColumn, callMonitorGetType( getActiveProfile() ) );

	psRenderer = gtk_cell_renderer_text_new();
	psDateTimeColumn = gtk_tree_view_column_new_with_attributes( _( "Date/Time" ), psRenderer, "text", COL_CM_DATETIME, NULL );
	g_object_set_data( G_OBJECT( psDateTimeColumn ), "model-column", GINT_TO_POINTER( COL_CM_DATETIME ) );
	g_signal_connect( G_OBJECT( psDateTimeColumn ), "clicked", G_CALLBACK( columnClicked ), psCallMonitorTreeView );
	gtk_tree_view_column_set_sort_column_id( psDateTimeColumn, COL_CM_DATETIME );
	gtk_tree_view_column_set_reorderable( psDateTimeColumn, TRUE );
	gtk_tree_view_column_set_resizable( psDateTimeColumn, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorTreeView ), psDateTimeColumn );
	gtk_tree_view_column_set_visible( psDateTimeColumn, callMonitorGetDateTime( getActiveProfile() ) );

	psRenderer = gtk_cell_renderer_text_new();
	psNameColumn = gtk_tree_view_column_new_with_attributes( _( "Name" ), psRenderer, "text", COL_CM_NAME, NULL );
	g_object_set_data( G_OBJECT( psNameColumn ), "model-column", GINT_TO_POINTER( COL_CM_NAME ) );
	g_signal_connect( G_OBJECT( psNameColumn ), "clicked", G_CALLBACK( columnClicked ), psCallMonitorTreeView );
	gtk_tree_view_column_set_sort_column_id( psNameColumn, COL_CM_NAME );
	gtk_tree_view_column_set_reorderable( psNameColumn, TRUE );
	gtk_tree_view_column_set_resizable( psNameColumn, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorTreeView ), psNameColumn );
	gtk_tree_view_column_set_visible( psNameColumn, callMonitorGetName( getActiveProfile() ) );

	/* Company column */
	psRenderer = gtk_cell_renderer_text_new();
	psCompanyColumn = gtk_tree_view_column_new_with_attributes( _( "Company" ), psRenderer, "text", COL_CM_COMPANY, NULL );
	g_object_set_data( G_OBJECT( psCompanyColumn ), "model-column", GINT_TO_POINTER( COL_CM_COMPANY ) );
	g_signal_connect( G_OBJECT( psCompanyColumn ), "clicked", G_CALLBACK( columnClicked ), psCallMonitorTreeView );
	gtk_tree_view_column_set_sort_column_id( psCompanyColumn, COL_CM_COMPANY );
	gtk_tree_view_column_set_reorderable( psCompanyColumn, TRUE );
	gtk_tree_view_column_set_resizable( psCompanyColumn, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorTreeView ), psCompanyColumn );
	gtk_tree_view_column_set_visible( psCompanyColumn, callMonitorGetCompany( getActiveProfile() ) );

	psRenderer = gtk_cell_renderer_text_new();
	psNumberColumn = gtk_tree_view_column_new_with_attributes( _( "Number" ), psRenderer, "text", COL_CM_NUMBER, NULL );
	g_object_set_data( G_OBJECT( psNumberColumn ), "model-column", GINT_TO_POINTER( COL_CM_NUMBER ) );
	g_signal_connect( G_OBJECT( psNumberColumn ), "clicked", G_CALLBACK( columnClicked ), psCallMonitorTreeView );
	gtk_tree_view_column_set_sort_column_id( psNumberColumn, COL_CM_NUMBER );
	gtk_tree_view_column_set_reorderable( psNumberColumn, TRUE );
	gtk_tree_view_column_set_resizable( psNumberColumn, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorTreeView ), psNumberColumn );
	gtk_tree_view_column_set_visible( psNumberColumn, callMonitorGetNumber( getActiveProfile() ) );

	/* City column */
	psRenderer = gtk_cell_renderer_text_new();
	psCityColumn = gtk_tree_view_column_new_with_attributes( _( "City" ), psRenderer, "text", COL_CM_CITY, NULL );
	g_object_set_data( G_OBJECT( psCityColumn ), "model-column", GINT_TO_POINTER( COL_CM_CITY ) );
	g_signal_connect( G_OBJECT( psCityColumn ), "clicked", G_CALLBACK( columnClicked ), psCallMonitorTreeView );
	gtk_tree_view_column_set_sort_column_id( psCityColumn, COL_CM_CITY );
	gtk_tree_view_column_set_reorderable( psCityColumn, TRUE );
	gtk_tree_view_column_set_resizable( psCityColumn, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorTreeView ), psCityColumn );
	gtk_tree_view_column_set_visible( psCityColumn, callMonitorGetCity( getActiveProfile() ) );
	gtk_tree_view_column_set_cell_data_func( psCityColumn, psRenderer, nameCellData, NULL, NULL );

	psRenderer = gtk_cell_renderer_text_new();
	psLocalNameColumn = gtk_tree_view_column_new_with_attributes( _( "Local Name" ), psRenderer, "text", COL_CM_LOCALNAME, NULL );
	g_object_set_data( G_OBJECT( psLocalNameColumn ), "model-column", GINT_TO_POINTER( COL_CM_LOCALNAME ) );
	g_signal_connect( G_OBJECT( psLocalNameColumn ), "clicked", G_CALLBACK( columnClicked ), psCallMonitorTreeView );
	gtk_tree_view_column_set_sort_column_id( psLocalNameColumn, COL_CM_LOCALNAME );
	gtk_tree_view_column_set_reorderable( psLocalNameColumn, TRUE );
	gtk_tree_view_column_set_resizable( psLocalNameColumn, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorTreeView ), psLocalNameColumn );
	gtk_tree_view_column_set_visible( psLocalNameColumn, callMonitorGetLocalName( getActiveProfile() ) );

	psRenderer = gtk_cell_renderer_text_new();
	psLocalNumberColumn = gtk_tree_view_column_new_with_attributes( _( "Local Number" ), psRenderer, "text", COL_CM_LOCALNUMBER, NULL );
	g_object_set_data( G_OBJECT( psLocalNumberColumn ), "model-column", GINT_TO_POINTER( COL_CM_LOCALNUMBER ) );
	g_signal_connect( G_OBJECT( psLocalNumberColumn ), "clicked", G_CALLBACK( columnClicked ), psCallMonitorTreeView );
	gtk_tree_view_column_set_sort_column_id( psLocalNumberColumn, COL_CM_LOCALNUMBER );
	gtk_tree_view_column_set_reorderable( psLocalNumberColumn, TRUE );
	gtk_tree_view_column_set_resizable( psLocalNumberColumn, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorTreeView ), psLocalNumberColumn );
	gtk_tree_view_column_set_visible( psLocalNumberColumn, callMonitorGetLocalNumber( getActiveProfile() ) );

	psRenderer = gtk_cell_renderer_text_new();
	psDurationColumn = gtk_tree_view_column_new_with_attributes( _( "Duration" ), psRenderer, "text", COL_CM_DURATION, NULL );
	g_object_set_data( G_OBJECT( psDurationColumn ), "model-column", GINT_TO_POINTER( COL_CM_DURATION ) );
	g_signal_connect( G_OBJECT( psDurationColumn ), "clicked", G_CALLBACK( columnClicked ), psCallMonitorTreeView );
	gtk_tree_view_column_set_sort_column_id( psDurationColumn, COL_CM_DURATION );
	gtk_tree_view_column_set_reorderable( psDurationColumn, TRUE );
	gtk_tree_view_column_set_resizable( psDurationColumn, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorTreeView ), psDurationColumn );
	gtk_tree_view_column_set_visible( psDurationColumn, callMonitorGetDuration( getActiveProfile() ) );

	psSelection = gtk_tree_view_get_selection( GTK_TREE_VIEW( psCallMonitorTreeView ) );
	gtk_tree_selection_set_mode( psSelection, GTK_SELECTION_MULTIPLE );

	gtk_tree_sortable_set_sort_func( psSortable, COL_CM_TYPE, sortByNumber, GINT_TO_POINTER( COL_CM_TYPE ), NULL );
	gtk_tree_sortable_set_sort_func( psSortable, COL_CM_DATETIME, sortByNumber, GINT_TO_POINTER( COL_CM_DATETIME ), NULL );
	gtk_tree_sortable_set_sort_func( psSortable, COL_CM_NAME, sortByNumber, GINT_TO_POINTER( COL_CM_NAME ), NULL );
	gtk_tree_sortable_set_sort_func( psSortable, COL_CM_COMPANY, sortByNumber, GINT_TO_POINTER( COL_CM_COMPANY ), NULL );
	gtk_tree_sortable_set_sort_func( psSortable, COL_CM_NUMBER, sortByNumber, GINT_TO_POINTER( COL_CM_NUMBER ), NULL );
	gtk_tree_sortable_set_sort_func( psSortable, COL_CM_CITY, sortByNumber, GINT_TO_POINTER( COL_CM_CITY ), NULL );
	gtk_tree_sortable_set_sort_func( psSortable, COL_CM_LOCALNAME, sortByNumber, GINT_TO_POINTER( COL_CM_LOCALNAME ), NULL );
	gtk_tree_sortable_set_sort_func( psSortable, COL_CM_LOCALNUMBER, sortByNumber, GINT_TO_POINTER( COL_CM_LOCALNUMBER ), NULL );
	gtk_tree_sortable_set_sort_func( psSortable, COL_CM_DURATION, sortByNumber, GINT_TO_POINTER( COL_CM_DURATION ), NULL );
	gtk_tree_sortable_set_sort_column_id( psSortable, COL_CM_DATETIME, GTK_SORT_DESCENDING );

	gtk_tree_view_set_search_column( GTK_TREE_VIEW( psCallMonitorTreeView ), COL_CM_NAME );

	psToolbar = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psToolbar" ) );

	switch ( miscGetToolbarStyle( getActiveProfile() ) ) {
		case 0:
			/* default */
			gtk_toolbar_unset_style( GTK_TOOLBAR( psToolbar ) );
			break;
		case 1:
			/* text below icons */
			gtk_toolbar_set_style( GTK_TOOLBAR( psToolbar ), GTK_TOOLBAR_BOTH );
			break;
		case 2:
			/* text beside icons */
			gtk_toolbar_set_style( GTK_TOOLBAR( psToolbar ), GTK_TOOLBAR_BOTH_HORIZ );
			break;
		case 3:
			/* icons only */
			gtk_toolbar_set_style( GTK_TOOLBAR( psToolbar ), GTK_TOOLBAR_ICONS );
			break;
		case 4:
			/* text only */
			gtk_toolbar_set_style( GTK_TOOLBAR( psToolbar ), GTK_TOOLBAR_TEXT );
			break;
		default:
			Debug( KERN_WARNING, "Unknown toolbar style, setting default\n" );
			gtk_toolbar_set_style( GTK_TOOLBAR( psToolbar ), GTK_TOOLBAR_TEXT );
			break;
	}

	psToolItem = gtk_tool_button_new_from_stock( GTK_STOCK_REFRESH );
	gtk_widget_set_tooltip_text( GTK_WIDGET( psToolItem ), _( "Refresh caller list from router" ) );
	g_signal_connect( G_OBJECT( psToolItem ), "clicked", G_CALLBACK( reloadCallMonitorStart ), psMonitorStore );
	gtk_toolbar_insert( GTK_TOOLBAR( psToolbar ), psToolItem, 0 );

	psAllProfilesItem = gtk_tool_button_new( NULL, _( "All profiles" ) );
	gtk_tool_button_set_stock_id( GTK_TOOL_BUTTON( psAllProfilesItem ), GTK_STOCK_DND_MULTIPLE );
	gtk_widget_set_tooltip_text( GTK_WIDGET( psAllProfilesItem ), _( "All profiles caller list" ) );
	g_signal_connect( G_OBJECT( psAllProfilesItem ), "clicked", G_CALLBACK( allProfiles ), psMonitorStore );
	gtk_toolbar_insert( GTK_TOOLBAR( psToolbar ), psAllProfilesItem, 1 );

	psNextProfileItem = gtk_tool_button_new( NULL, _( "Next profile" ) );
	gtk_tool_button_set_stock_id( GTK_TOOL_BUTTON( psNextProfileItem ), GTK_STOCK_DND );
	gtk_widget_set_tooltip_text( GTK_WIDGET( psNextProfileItem ), _( "Next profile caller list" ) );
	g_signal_connect( G_OBJECT( psNextProfileItem ), "clicked", G_CALLBACK( nextProfile ), psMonitorStore );
	gtk_toolbar_insert( GTK_TOOLBAR( psToolbar ), psNextProfileItem, 2 );

	psAddToolItem = gtk_tool_button_new_from_stock( GTK_STOCK_ADD );
	gtk_widget_set_tooltip_text( GTK_WIDGET( psAddToolItem ), _( "Add selected contact" ) );
	g_signal_connect( G_OBJECT( psAddToolItem ), "clicked", G_CALLBACK( callMonitorAdd ), psCallMonitorTreeView );
	gtk_toolbar_insert( GTK_TOOLBAR( psToolbar ), psAddToolItem, 3 );

	psBook = getDefaultBookPlugin( getActiveProfile() );
	if ( psBook == NULL || psBook -> SaveBook == NULL ) {
		gtk_widget_set_sensitive( GTK_WIDGET( psAddToolItem ), FALSE );
	}

	if ( psProfiles == NULL || g_list_length( psProfiles ) < 2 ) {
		gtk_widget_set_sensitive( GTK_WIDGET( psAllProfilesItem ), FALSE );
		gtk_widget_set_sensitive( GTK_WIDGET( psNextProfileItem ), FALSE );
	} else {
		gtk_widget_set_sensitive( GTK_WIDGET( psAllProfilesItem ), TRUE );
		gtk_widget_set_sensitive( GTK_WIDGET( psNextProfileItem ), TRUE );
	}

	psToolItem = gtk_tool_button_new_from_stock( GTK_STOCK_DELETE );
	gtk_widget_set_tooltip_text( GTK_WIDGET( psToolItem ), _( "Delete selected entry" ) );
	g_signal_connect( G_OBJECT( psToolItem ), "clicked", G_CALLBACK( callMonitorDelete ), psCallMonitorTreeView );
	gtk_toolbar_insert( GTK_TOOLBAR( psToolbar ), psToolItem, 4 );

	psToolItem = gtk_tool_button_new( NULL, _( "Reverse lookup" ) );
	gtk_tool_button_set_stock_id( GTK_TOOL_BUTTON( psToolItem ), GTK_STOCK_FIND );
	gtk_widget_set_tooltip_text( GTK_WIDGET( psToolItem ), _( "Reverse lookup of selected contact" ) );
	g_signal_connect( G_OBJECT( psToolItem ), "clicked", G_CALLBACK( callMonitorFind ), psCallMonitorTreeView );
	gtk_toolbar_insert( GTK_TOOLBAR( psToolbar ), psToolItem, 5 );

	psToolItem = gtk_tool_button_new_from_stock( GTK_STOCK_CLEAR );
	gtk_widget_set_tooltip_text( GTK_WIDGET( psToolItem ), _( "Delete caller list on router" ) );
	g_signal_connect( G_OBJECT( psToolItem ), "clicked", G_CALLBACK( deleteCallMonitor ), NULL );
	gtk_toolbar_insert( GTK_TOOLBAR( psToolbar ), psToolItem, 6 );

	psToolItem = gtk_tool_button_new_from_stock( GTK_STOCK_PRINT );
	gtk_widget_set_tooltip_text( GTK_WIDGET( psToolItem ), _( "Print caller list" ) );
	g_signal_connect( G_OBJECT( psToolItem ), "clicked", G_CALLBACK( PrintTreeView ), psCallMonitorTreeView );
	gtk_toolbar_insert( GTK_TOOLBAR( psToolbar ), psToolItem, 7 );

	/* Lookup plugin */
	psLookupComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psLookupComboBox" ) );
	psModel = gtk_combo_box_get_model( GTK_COMBO_BOX( psLookupComboBox ) );
	psLookupStore = GTK_LIST_STORE( psModel );
	gtk_list_store_clear( GTK_LIST_STORE( psLookupStore ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( psLookupComboBox ), _( "None" ) );
	gtk_combo_box_set_active( GTK_COMBO_BOX( psLookupComboBox ), nCount );

	for ( psList = getLookupList( routerGetCountryCode( getActiveProfile() ) ); psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		psLookup = psList -> data;
		gtk_combo_box_append_text( GTK_COMBO_BOX( psLookupComboBox ), psLookup -> pnService );
		nCount++;
		if ( strcmp( psLookup -> pnService, pluginsGetLookup( getActiveProfile() ) ) == 0 ) {
			gtk_combo_box_set_active( GTK_COMBO_BOX( psLookupComboBox ), nCount );
		}
	}
	g_signal_connect( G_OBJECT( psLookupComboBox ), "changed", G_CALLBACK( callMonitorLookupChanged ), NULL );

	g_signal_connect( G_OBJECT( psCallMonitor ), "configure_event", G_CALLBACK( callMonitorConfigure ), NULL );
	g_signal_connect( G_OBJECT( psCallMonitor ), "delete-event", G_CALLBACK( callMonitorResponse ), NULL );

	gtk_builder_connect_signals( psBuilder, NULL );

	g_object_unref( G_OBJECT( psBuilder ) );

	updateFilter( psFilterStore, psSelectionTreeView );

	callMonitorGetSize( getActiveProfile(), &nCallMonitorWidth, &nCallMonitorHeight );

	if ( gtk_tree_model_get_iter_first( GTK_TREE_MODEL( psListStore ), &sIter ) == TRUE ) {
		gtk_tree_selection_select_iter( gtk_tree_view_get_selection( GTK_TREE_VIEW( psCallMonitorTreeView ) ), &sIter );
	}

	gtk_window_set_default_size( GTK_WINDOW( psCallMonitor ), nCallMonitorWidth, nCallMonitorHeight );

	pnDummy = getProfileCaption();
	gtk_window_set_title( GTK_WINDOW( psCallMonitor ), pnDummy );
	g_free( pnDummy );

	g_signal_connect( G_OBJECT( psCallMonitorTreeView ), "button-press-event", G_CALLBACK( callMonitorPressed ), psMonitorStore );

	gtk_widget_show_all( psCallMonitor );

	gint nPosition;
	nPosition = callMonitorGetPanedPosition( getActiveProfile() );
	gtk_paned_set_position( GTK_PANED( psPaned ), nPosition );

	return psCallMonitor;
}
