require_relative '../../spec_helper'
require_relative 'spec_helper'
require_relative 'fixtures/server'

describe "Net::FTP#acct" do
  before :each do
    @server = NetFTPSpecs::DummyFTP.new
    @server.serve_once

    @ftp = Net::FTP.new
    @ftp.connect(@server.hostname, @server.server_port)
  end

  after :each do
    @ftp.quit rescue nil
    @ftp.close
    @server.stop
  end

  it "writes the ACCT command to the server" do
    @ftp.acct("my_account")
    @ftp.last_response.should == "230 User 'my_account' logged in, proceed. (ACCT)\n"
  end

  it "returns nil" do
    @ftp.acct("my_account").should == nil
  end

  it "does not raise any error when the response code is 230" do
    @server.should_receive(:acct).and_respond("230 User logged in, proceed.")
    -> { @ftp.acct("my_account") }.should_not raise_error
  end

  it "raises a Net::FTPPermError when the response code is 530" do
    @server.should_receive(:acct).and_respond("530 Not logged in.")
    -> { @ftp.acct("my_account") }.should raise_error(Net::FTPPermError)
  end

  it "raises a Net::FTPPermError when the response code is 500" do
    @server.should_receive(:acct).and_respond("500 Syntax error, command unrecognized.")
    -> { @ftp.acct("my_account") }.should raise_error(Net::FTPPermError)
  end

  it "raises a Net::FTPPermError when the response code is 501" do
    @server.should_receive(:acct).and_respond("501 Syntax error in parameters or arguments.")
    -> { @ftp.acct("my_account") }.should raise_error(Net::FTPPermError)
  end

  it "raises a Net::FTPPermError when the response code is 503" do
    @server.should_receive(:acct).and_respond("503 Bad sequence of commands.")
    -> { @ftp.acct("my_account") }.should raise_error(Net::FTPPermError)
  end

  it "raises a Net::FTPTempError when the response code is 421" do
    @server.should_receive(:acct).and_respond("421 Service not available, closing control connection.")
    -> { @ftp.acct("my_account") }.should raise_error(Net::FTPTempError)
  end
end
