/***********************************************************************************

    Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "diarydata.hpp"
#include "lifeograph.hpp"

#include <cassert>


using namespace LIFEO;


bool
LIFEO::compare_listitems( DiaryElement *item_l, DiaryElement *item_r )
{
    return( item_l->get_date() > item_r->get_date() );
}

bool
LIFEO::compare_listitems_by_name( DiaryElement *item_l, DiaryElement *item_r )
{
    return( item_l->get_name() < item_r->get_name() );
}

bool
LIFEO::compare_names( const Glib::ustring &name_l, const Glib::ustring &name_r )
{
    return( name_l < name_r );
}

// DIARYELEMENT ====================================================================================
// STATIC MEMBERS
ListData::Colrec                    *ListData::colrec;
bool                                DiaryElement::FLAG_ALLOCATE_GUI_FOR_DIARY = true;
const Glib::RefPtr< Gdk::Pixbuf >   DiaryElement::s_pixbuf_null;

DiaryElement::DiaryElement( DiaryBase * const ptr2diary, const Glib::ustring &name )
:   NamedElement( name ), m_ptr2diary( ptr2diary ),
    m_id( ptr2diary ? ptr2diary->create_new_id( this ) : DEID_UNSET )
{
    if( FLAG_ALLOCATE_GUI_FOR_DIARY )
        m_list_data = new ListData;
}


// TAG =============================================================================================
// STATIC MEMBERS
ElementShower< Tag >    *Tag::shower( NULL );

Tag::Tag( DiaryBase * const d, const Glib::ustring &name, CategoryTags *category )
:   DiaryElementContainer( d, name ), m_ptr2category( category )
{
    if( category != NULL )
        category->insert( this );
}

Tag::~Tag( void )
{
}

const Glib::RefPtr< Gdk::Pixbuf >&
Tag::get_icon( void ) const
{
    return( this == Diary::d->get_filter_tag() ?
            Lifeograph::icons->filter_16 : Lifeograph::icons->tag_16 );
}
const Glib::RefPtr< Gdk::Pixbuf >&
Tag::get_icon32( void ) const
{
    return( Lifeograph::icons->tag_32 );
}

void
Tag::show( void )
{
    if( shower != NULL )
        shower->show( *this );
    else
        PRINT_DEBUG( "Tag has no graphical data!" );
}

void
Tag::set_name( const Glib::ustring &name )
{
    m_name = name;
}

void
Tag::set_category( CategoryTags *category_new )
{
    if( m_ptr2category )
        m_ptr2category->erase( this );
    if( category_new )
        category_new->insert( this );
    m_ptr2category = category_new;
}

// TAGPOOL =========================================================================================
PoolTags::~PoolTags()
{
    for( iterator iter = begin(); iter != end(); ++iter )
        delete iter->second;
}

Tag*
PoolTags::get_tag( unsigned int tagorder )
{
    if( tagorder >= size() )
        return NULL;

    const_iterator iter = begin();
    advance( iter, tagorder );
    return iter->second;
}

Tag*
PoolTags::get_tag( const Glib::ustring &name )
{
    iterator iter = find( name );
    if( iter != end() )
        return( iter->second );
    else
        return NULL;
}

bool
PoolTags::rename( Tag *tag, const Glib::ustring &new_name )
{
    erase( tag->m_name );
    tag->m_name = new_name;
    return( insert( value_type( new_name, tag ) ).second );
}

void
PoolTags::clear( void )
{
    for( iterator iter = begin(); iter != end(); ++iter )
        delete iter->second;

    std::map< Glib::ustring, Tag*, FuncCompareStrings >::clear();
}

// CATEGORYTAGS ====================================================================================
// STATIC MEMBERS
ElementShower< CategoryTags >   *CategoryTags::shower( NULL );

CategoryTags::CategoryTags( DiaryBase * const d, const Glib::ustring &name )
:   DiaryElement( d, name ), std::set< Tag*, FuncCompareDiaryElemNamed >( compare_listitems_by_name ),
    m_flag_expanded( true )
{
}

const Glib::RefPtr< Gdk::Pixbuf >&
CategoryTags::get_icon( void ) const
{
    return Lifeograph::icons->tag_category_16;
}
const Glib::RefPtr< Gdk::Pixbuf >&
CategoryTags::get_icon32( void ) const
{
    return Lifeograph::icons->tag_category_32;
}

void
CategoryTags::show( void )
{
    if( shower != NULL )
        shower->show( *this );
    else
        PRINT_DEBUG( "Category has no graphical data!" );
}

// POOL OF DEFINED TAG CATEGORIES
PoolCategoriesTags::PoolCategoriesTags( void )
:   std::map< Glib::ustring, CategoryTags*, FuncCompareStrings >( compare_names )
{
}

PoolCategoriesTags::~PoolCategoriesTags()
{
    for( iterator iter = begin(); iter != end(); ++iter )
    {
        delete iter->second;
    }
}

void
PoolCategoriesTags::clear( void )
{
    for( iterator iter = begin(); iter != end(); ++iter )
    {
        delete iter->second;
    }

    std::map< Glib::ustring, CategoryTags*, FuncCompareStrings >::clear();
}

bool
PoolCategoriesTags::rename_category( CategoryTags *category, const Glib::ustring &new_name )
{
    erase( category->m_name );
    category->m_name = new_name;
    return( insert( value_type( new_name, category ) ).second );
}

// TAGSET
Tagset::~Tagset( void )
{
}

bool
Tagset::add( Tag *tag )
{
    if( insert( tag ).second ) // if did not exist previously
        return true;
    else
    {
        PRINT_DEBUG( " tagset already has the tag " + tag->get_name() );
        return false;
    }

}

bool
Tagset::checkfor_member( const Tag *tag ) const
{
    return( count( const_cast< Tag* >( tag ) ) > 0 );
}

const Tag*
Tagset::get_tag( unsigned int tagorder ) const
{
    unsigned int i = 0;

    for( Tagset::const_iterator iter = this->begin(); iter != this->end(); ++iter )
    {
        if( i == tagorder )
        {
            return( *iter );
        }
        i++;
    }
    return NULL;
}

// CHAPTER =========================================================================================
// STATIC MEMBERS
ElementShower< Chapter >    *Chapter::shower( NULL );

Chapter::Chapter( DiaryBase * const d, const Glib::ustring &name )
:   DiaryElement( d, name ), m_date_begin( Date::NOTSET ), m_time_span( 0 ), m_size( 0 ),
    m_flag_expanded( false )
{
}

Chapter::Chapter( DiaryBase * const d, const Glib::ustring &name, Date date )
    :   DiaryElement( d, name ), m_date_begin( date ), m_size( 0 ),
    m_flag_expanded( false )
{
}

const Glib::RefPtr< Gdk::Pixbuf >&
Chapter::get_icon( void ) const
{
    return m_date_begin.is_ordinal() ?
                Lifeograph::icons->chapter_16 : Lifeograph::icons->month_16;
}
const Glib::RefPtr< Gdk::Pixbuf >&
Chapter::get_icon32( void ) const
{
    return m_date_begin.is_ordinal() ?
                Lifeograph::icons->chapter_32 : Lifeograph::icons->month_32;
}

void
Chapter::show( void )
{
    if( shower != NULL )
        shower->show( *this );
    else
        PRINT_DEBUG( "Chapter has no graphical data!" );

}

void
Chapter::set_name( const Glib::ustring &name )
{
    m_name = name;
}

Glib::ustring
Chapter::get_date_str( void ) const
{
    if( is_initialized() )
        return m_date_begin.format_string();
    else
        return "";
}

Date
Chapter::get_date( void ) const
{
    return m_date_begin;
}

void
Chapter::set_date( Date::date_t date )
{
    m_date_begin.m_date = date;
}

Date
Chapter::get_free_order( void ) const
{
    Date date( m_date_begin );
    m_ptr2diary->make_free_entry_order( date );
    return date;
}

void
Chapter::recalculate_span( const Chapter *next )
{
    if( next == NULL )
        m_time_span = 0;    // unlimited
    else if( next->m_date_begin.is_ordinal() )
        m_time_span = 0;    // last temporal chapter: unlimited
    else
        m_time_span = m_date_begin.calculate_days_between( next->m_date_begin );
}

// CHAPTER CATEGORY ================================================================================
CategoryChapters::CategoryChapters( DiaryBase * const d, const Glib::ustring &name )
:   DiaryElement( d, name ),
    std::map< Date::date_t, Chapter*, FuncCompareDates >( compare_dates )
{
}

CategoryChapters::~CategoryChapters()
{
    for( CategoryChapters::iterator iter = begin(); iter != end(); ++iter )
        delete iter->second;
}

Chapter*
CategoryChapters::get_chapter( const Date::date_t date ) const
{
    const_iterator iter( find( date ) );
    return( iter == end() ? NULL : iter->second );
}

Chapter*
CategoryChapters::create_chapter( const Glib::ustring &name, const Date &date )
{
    Chapter *chapter = new Chapter( m_ptr2diary, name, date );

    add( chapter );

    return chapter;
}

void
CategoryChapters::dismiss_chapter( Chapter *chapter )
{
    iterator iter( find( chapter->m_date_begin.m_date ) );
    if( iter == end() )
    {
        print_error( "chapter could not be found in assumed category" );
        return;
    }
    else if( ( ++iter ) != end() )  // fix time span
    {
        Chapter *chapter_earlier( iter->second );
        if( chapter->m_time_span > 0 )
            chapter_earlier->m_time_span += chapter->m_time_span;
        else
            chapter_earlier->m_time_span = 0;
    }
    erase( chapter->m_date_begin.m_date );
    delete chapter;
}

bool
CategoryChapters::rename_chapter( Chapter *chapter, const Glib::ustring &name )
{
    // Chapters must have unique begin dates not names
    chapter->set_name( name );
    return true;
}

bool
CategoryChapters::set_chapter_date( Chapter *chapter, Date::date_t date )
{
    assert( chapter->is_ordinal() == false );

    if( chapter->m_date_begin.m_date != Date::NOTSET )
    {
        iterator iter( find( chapter->m_date_begin.m_date ) );
        if( iter == end() )
            return false; // chapter is not a member of the set

        if( ( ++iter ) != end() ) // fix time span
        {
            Chapter *chapter_earlier( iter->second );
            if( chapter->m_time_span > 0 )
                chapter_earlier->m_time_span += chapter->m_time_span;
            else
                chapter_earlier->m_time_span = 0;
        }

        erase( chapter->m_date_begin.m_date );

        if( find( date ) != end() ) // if target is taken move chapters
        {
            Date::date_t d( chapter->m_date_begin.is_ordinal() ?
                            chapter->m_date_begin.m_date :
                            get_free_order_ordinal().m_date );
                            // if a temporal chapter is being converted..
                            // ..assume that it was the last
            Date::date_t step( chapter->m_date_begin.m_date > date ?
                                        -Date::ORDINAL_STEP : Date::ORDINAL_STEP );

            for( d += step; ; d += step )
            {
                iter = find( d );
                if( iter == end() )
                    break;

                Chapter *chapter_shift( iter->second );
                erase( d );
                chapter_shift->set_date( d - step );
                insert( CategoryChapters::value_type( chapter_shift->m_date_begin.m_date,
                                                      chapter_shift ) );

                if( d == date )
                    break;
            }
        }
    }

    chapter->set_date( date );

    add( chapter );

    return true;
}

bool
CategoryChapters::add( Chapter *chapter )
{
    iterator iter = insert( CategoryChapters::value_type(
            chapter->m_date_begin.m_date, chapter ) ).first;

    if( iter == begin() ) // latest
        chapter->recalculate_span( NULL );
    else
    {
        iterator iter_next( iter );
        iter_next--;
        chapter->recalculate_span( iter_next->second );
    }

    if( ( ++iter ) != end() ) // if not earliest fix previous
        iter->second->recalculate_span( chapter );

    return true; // reserved
}

void
CategoryChapters::clear( void )
{
    for( CategoryChapters::iterator iter = begin(); iter != end(); ++iter )
        delete iter->second;

    std::map< Date::date_t, Chapter*, FuncCompareDates >::clear();
}

Date
CategoryChapters::get_free_order_ordinal( void ) const
{
    if( size() > 0 )
    {
        Date d( begin()->first );
        if( d.is_ordinal() )
        {
            d.forward_ordinal_order();
            return d;
        }
    }

    return Date( 0, 0 );
}

PoolCategoriesChapters::~PoolCategoriesChapters()
{
    for( PoolCategoriesChapters::iterator iter = begin(); iter != end(); ++iter )
        delete( iter->second );
}

void
PoolCategoriesChapters::clear( void )
{
    for( PoolCategoriesChapters::iterator iter = begin(); iter != end(); ++iter )
        delete( iter->second );

    std::map< Glib::ustring, CategoryChapters*, FuncCompareStrings >::clear();
}

// THEMES ==========================================================================================
// STATIC MEMBERS
ElementShower< Theme > *Theme::shower( NULL );
const Gdk::RGBA Theme::s_color_match( "#33FF33" );
const Gdk::RGBA Theme::s_color_match2( "#009900" );
const Gdk::RGBA Theme::s_color_link( "#3333FF" );
const Gdk::RGBA Theme::s_color_link2( "#000099" );
const Gdk::RGBA Theme::s_color_broken( "#FF3333" );
const Gdk::RGBA Theme::s_color_broken2( "#990000" );

Theme::Theme( DiaryBase * const d, const Glib::ustring &name )
:   DiaryElement( d, name ),
    font( ThemeSystem::get()->font ),
    color_base( ThemeSystem::get()->color_base ),
    color_text( ThemeSystem::get()->color_text ),
    color_heading( ThemeSystem::get()->color_heading ),
    color_subheading( ThemeSystem::get()->color_subheading ),
    color_highlight( ThemeSystem::get()->color_highlight )
{
}

Theme::Theme( DiaryBase * const d,
              const Glib::ustring &name,
              const Glib::ustring &str_font,
              const std::string &str_base,
              const std::string &str_text,
              const std::string &str_heading,
              const std::string &str_subheading,
              const std::string &str_highlight )
:   DiaryElement( d, name ), font( str_font ), color_base( str_base ), color_text( str_text ),
    color_heading( str_heading ), color_subheading( str_subheading ),
    color_highlight( str_highlight )
{
}

Theme::Theme( DiaryBase * const d, const Theme *theme, const Glib::ustring &name )
:   DiaryElement( d, name ),
    font( theme->font ),
    color_base( theme->color_base ),
    color_text( theme->color_text ),
    color_heading( theme->color_heading ),
    color_subheading( theme->color_subheading ),
    color_highlight( theme->color_highlight )
{
}

const Glib::RefPtr< Gdk::Pixbuf >&
Theme::get_icon( void ) const
{
    return( this == Diary::d->get_default_theme() ?
            Lifeograph::icons->theme_default_16 : Lifeograph::icons->theme_16 );
}
const Glib::RefPtr< Gdk::Pixbuf >&
Theme::get_icon32( void ) const
{
    return( Lifeograph::icons->theme_32 );
}

void
Theme::show( void )
{
    if( shower != NULL )
        shower->show( *this );
    else
        PRINT_DEBUG( "Theme has no graphical data!" );
}

ThemeSystem::ThemeSystem( const Glib::ustring &f,
                          const std::string &cb,
                          const std::string &ct,
                          const std::string &ch,
                          const std::string &csh,
                          const std::string &chl )
:   Theme( NULL, STRING::SYSTEM_THEME, f, cb, ct, ch, csh, chl )
{
}

ThemeSystem*
ThemeSystem::get( void )
{
    static bool s_flag_initialized( false );
    static ThemeSystem *s_theme;

    if( ! s_flag_initialized )
    {
        // this may not be the best method to detect the default font:
        Gtk::TextView *tv = new Gtk::TextView;
        tv->show();
        s_theme = new ThemeSystem( tv->get_style_context()->get_font().to_string(),
                                   "white", "black",
                                   "blue", "#F066FC",
                                   "#FFF955" );
        delete tv;
        s_flag_initialized = true;
        return s_theme;
    }
    else
    {
        return s_theme;
    }
}

// POOLTHEMES ==================================================================
PoolThemes::~PoolThemes()
{
    for( iterator iter = begin(); iter != end(); ++iter )
        if( ! iter->second->is_system() )
            delete iter->second;
}

void
PoolThemes::clear( void )
{
    for( iterator iter = begin(); iter != end(); ++iter )
        if( iter->second->is_system() == false )
            delete iter->second;

    std::map< Glib::ustring, Theme*, FuncCompareStrings >::clear();
    // add back the system theme:
    insert( PoolThemes::value_type( STRING::SYSTEM_THEME, ThemeSystem::get() ) );
}

bool
PoolThemes::rename_theme( Theme *theme, const Glib::ustring &new_name )
{
    erase( theme->get_name() );
    theme->set_name( new_name );
    return( insert( value_type( new_name, theme ) ).second );
}
