///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file AnalyzeMicrostructureModifier.h
 * \brief Contains the definition of the CrystalAnalysis::AnalyzeMicrostructureModifier class.
 */

#ifndef __ANALYZE_MICROSTRUCTURE_MODIFIER_H
#define __ANALYZE_MICROSTRUCTURE_MODIFIER_H

#include <core/Core.h>
#include <core/gui/properties/PropertiesEditor.h>

#include "../CrystalAnalysis.h"
#include <atomviz/modifier/AtomsObjectAnalyzerBase.h>
#include "FindGrains.h"

namespace CrystalAnalysis {

/**
 * Analyze a nanocrystalline structure with respect to several features.
 *
 * Currently only the decomposition algorithm to find individual grains is implemented.
 *
 * \author Alexander Stukowski
 */
class CRYSTALANALYSIS_DLLEXPORT AnalyzeMicrostructureModifier : public AtomsObjectAnalyzerBase
{
public:

	/// \brief Default constructor.
	/// \param isLoading Indicates whether the object is being loaded from a file.
	///                  This parameter is only used by the object serialization system.
	AnalyzeMicrostructureModifier(bool isLoading = false);

public:

	/// Performs the analysis.
	/// Throws an exception on error.
	/// Returns false when the operation has been canceled by the user.
	bool calculate(AtomsObject* atomsObject, bool suppressDialogs = false);

	/// Returns the internal data channel that stores the computed grain cluster numbers for each atom.
	DataChannel* clusterChannel() const { return _clusterChannel; }

	/// Returns the internal data channel that stores the computed atomic misorientation
	DataChannel* misorientationChannel() const { return _misorientationChannel; }

	/// \brief Sets the threshold angle (in radians) for grain identification.
	///
	/// If the orientation of two neighboring atom clusters differs by less than the threshold
	/// angle then they are joined to a single grain.
	///
	/// \undoable
	/// \sa grainMisorientationThreshold()
	void setGrainMisorientationThreshold(FloatType grainMisorientationThreshold) { _grainMisorientationThreshold = grainMisorientationThreshold; }

	/// \brief Returns the current threshold angle (in radians) for grain identification.
	/// \sa setGrainMisorientationThreshold()
	int grainMisorientationThreshold() const { return _grainMisorientationThreshold; }

	/// \brief Sets the minimum size (=number of crystalline atoms) a cluster of atoms has
	///        to reach to be considered a full grain.
	/// \undoable
	/// \sa minimumGrainSize()
	void setMinimumGrainSize(int minimumGrainSize) { _minimumGrainSize = minimumGrainSize; }

	/// \brief Returns the minimum size (=number of crystalline atoms) a cluster of atoms has
	///        to reach to be considered a full grain.
	/// \sa setMinimumGrainSize()
	int minimumGrainSize() const { return _minimumGrainSize; }

	/// \brief Sets whether a random color is assigned to all atoms of an identified grain.
	/// \undoable
	/// \sa assignRandomColors()
	void setAssignRandomColors(bool flag) { _assignRandomColors = flag; }

	/// \brief Returns whether a random color is assigned to all atoms of an identified grain.
	/// \sa setAssignRandomColors()
	bool assignRandomColors() const { return _assignRandomColors; }

	/// \brief Controls whether the atomic misorientation angles are computed and stored in the output channel.
	/// \undoable
	/// \sa calculateAtomicMisorientations()
	/// \sa misorientationChannel()
	void setCalculateAtomicMisorientations(bool flag) { _calculateAtomicMisorientations = flag; }

	/// \brief Returns whether the atomic misorientation angles are computed and stored in the output channel.
	/// \sa setCalculateAtomicMisorientations()
	/// \sa misorientationChannel()
	bool calculateAtomicMisorientations() const { return _calculateAtomicMisorientations; }

	/// \brief Returns the list of identified grains.
	const QVector<FindGrains::GrainInfo>& grains() const { return _findGrains.grains(); }

	/// \brief Returns the list of identified grain boundaries.
	const QVector<FindGrains::GrainBoundaryInfo>& grainBoundaries() const { return _findGrains.grainBoundaries(); }

public:

	Q_PROPERTY(FloatType grainMisorientationThreshold READ grainMisorientationThreshold WRITE setGrainMisorientationThreshold)
	Q_PROPERTY(int minimumGrainSize READ minimumGrainSize WRITE setMinimumGrainSize)
	Q_PROPERTY(bool assignRandomColors READ assignRandomColors WRITE setAssignRandomColors)
	Q_PROPERTY(bool calculateAtomicMisorientations READ calculateAtomicMisorientations WRITE setCalculateAtomicMisorientations)

protected:

	// RefTarget virtual functions:
	virtual void saveToStream(ObjectSaveStream& stream);
	virtual void loadFromStream(ObjectLoadStream& stream);
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

	/// This is the actual analysis method.
	/// It is responsible for storing the analysis results in the results container object that
	/// will be stored along with the modifier application object.
	virtual EvaluationStatus doAnalysis(TimeTicks time, bool suppressDialogs);

	/// Applies the previously calculated analysis results to the atoms object.
	virtual EvaluationStatus applyResult(TimeTicks time, TimeInterval& validityInterval);

	/// This output data channel stores the grain cluster numbers for each atom.
	ReferenceField<DataChannel> _clusterChannel;

	/// This output data channel stores the atomic misorientation angles
	ReferenceField<DataChannel> _misorientationChannel;

	/// The helper object that identifies individual grains.
	FindGrains _findGrains;

	/// This stores the misorientation threshold angle for grain decomposition.
	PropertyField<FloatType> _grainMisorientationThreshold;

	/// This stores the minimum number of crystalline atoms for a cluster to be considered a grain.
	PropertyField<int> _minimumGrainSize;

	/// Controls the generation of random colors.
	PropertyField<bool> _assignRandomColors;

	/// Controls the computation of atomic misorientation angles.
	PropertyField<bool> _calculateAtomicMisorientations;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(AnalyzeMicrostructureModifier)
	DECLARE_REFERENCE_FIELD(_clusterChannel)
	DECLARE_REFERENCE_FIELD(_misorientationChannel)
	DECLARE_PROPERTY_FIELD(_grainMisorientationThreshold)
	DECLARE_PROPERTY_FIELD(_minimumGrainSize)
	DECLARE_PROPERTY_FIELD(_assignRandomColors)
	DECLARE_PROPERTY_FIELD(_calculateAtomicMisorientations)
};

/**
 * \brief A properties editor for the AnalyzeMicrostructureModifier class.
 *
 * This editor class creates and manages the user interface through which the
 * user can alter the modifier's parameters.
 *
 * \author Alexander Stukowski
 * \sa AnalyzeMicrostructureModifier
 */
class CRYSTALANALYSIS_DLLEXPORT AnalyzeMicrostructureModifierEditor : public AtomsObjectModifierEditorBase
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

protected Q_SLOTS:

	/// Is called when the user presses the Recalculate button.
	void onRecalculate();

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(AnalyzeMicrostructureModifierEditor)
};

};	// End of namespace CrystalAnalysis

#endif // __ANALYZE_MICROSTRUCTURE_MODIFIER_H
