/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment image wrapping example
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#include <pgm/pgm.h>

static PgmViewport *viewport = NULL;

/* Transform the mapping matrix of the image */
static gboolean
update_pass_cb (PgmViewport *viewport,
                gpointer data)
{
  PgmImage *image = PGM_IMAGE (data);
  static PgmMat4x4 *mapping = NULL;

  /* Get the default mapping matrix at the first call and apply a scale on the
   * x axis to correct the aspect-ratio of the image content distorted by the
   * filled layout and the size of the image */
  if (G_UNLIKELY (!mapping))
    {
      pgm_image_get_mapping_matrix (PGM_IMAGE (image), &mapping);
      pgm_mat4x4_scale_from_scalars (mapping, 800.0f / 64.0f, 1.0f, 0.0f);
    }

  /* Simply translate the matrix on x to apply the effect */
  pgm_mat4x4_translate_from_scalars (mapping, -0.005f, 0.0f, 0.0f);

  /* Set the new mapping matrix */
  pgm_image_set_mapping_matrix (PGM_IMAGE (image), mapping);

  return TRUE;
}

/* key-press handler */
static void
keypress_event_cb (PgmViewport *viewport,
                   PgmEventKey *event,
                   gpointer data)
{
  static gboolean decorated = TRUE;

  if (event->type == PGM_KEY_PRESS)
    {
      switch (event->keyval)
        {
          /* Add/remove window decorations */
        case PGM_d:
          decorated = !decorated;
          pgm_viewport_set_decorated (viewport, decorated);
          break;

          /* Handle quit */
        case PGM_q:
        case PGM_Escape:
          pgm_main_quit ();
          break;

        default:
          break;
        }
    }
}

/* Entry point */
int
main (int argc,
      char *argv[])
{
  const gchar *img_path = "examples/pictures/line-pattern.png";
  PgmDrawable *img;
  PgmCanvas *canvas;

  /* Init */
  pgm_init (&argc, &argv);

  /* OpenGL viewport creation */
  pgm_viewport_factory_make ("opengl", &viewport);
  if (!viewport)
    {
      g_print ("Cannot create the 'opengl' viewport\n");
      return -1;
    }
  pgm_viewport_set_size (viewport, 800, 164);
  pgm_viewport_set_title (viewport, "Image wrapping");

  /* Image definition */
  img = pgm_image_new_from_file (img_path, 0);
  pgm_image_set_layout (PGM_IMAGE (img), PGM_IMAGE_FILLED);
  pgm_image_set_wrapping (PGM_IMAGE (img), PGM_IMAGE_REPEAT, PGM_IMAGE_CLAMP);
  pgm_drawable_set_position (img, 50, 50, 0);
  pgm_drawable_set_size (img, 700, 64);
  pgm_drawable_set_bg_color (img, 0, 0, 0, 0);
  pgm_drawable_show (img);

  /* Canvas handling */
  canvas = pgm_canvas_new ();
  pgm_canvas_set_size (canvas, 800, 164);
  pgm_viewport_set_canvas (viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img);

  /* Main loop */
  g_signal_connect (G_OBJECT (viewport), "key-press-event",
                    G_CALLBACK (keypress_event_cb), NULL);
  g_signal_connect (G_OBJECT (viewport), "delete-event",
                    G_CALLBACK (pgm_main_quit), NULL);
  g_signal_connect (G_OBJECT (viewport), "update-pass",
                    G_CALLBACK (update_pass_cb), img);
  pgm_viewport_show (viewport);

  pgm_main ();

  /* Deinit */
  gst_object_unref (canvas);
  gst_object_unref (viewport);
  pgm_deinit ();

  return 0;
}
