/*==============================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*/
#include <align/extern.h>

#include <klib/log.h>
#include <klib/rc.h>
#include <kapp/main.h>
#include <klib/sort.h>
#include <klib/container.h>
#include <kfs/mmap.h>
#include <kfs/file.h>
#include <kdb/manager.h>
#include <vdb/database.h>
#include <vdb/table.h>
#include <vdb/manager.h>
#include <sra/sradb.h>

#include <align/writer-reference.h>
#include <align/writer-refseq.h>
#include <align/refseq-mgr.h>
#include "refseq-mgr-priv.h"
#include "writer-ref.h"
#include "reader-cmn.h"
#include "debug.h"
#include <os-native.h>
#include <sysalloc.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

struct ReferenceMgr {
    const TableWriterRef* writer;
    const KDirectory* dir;
    BSTree tree;
    uint32_t options;
    const VDBManager* vmgr;
    const RefSeqMgr* rmgr;
    VDatabase* db;
    size_t cache;
    uint32_t num_open_max;
    uint32_t num_open;
    uint64_t usage;
    uint32_t max_seq_len;
    /* last seq end */
    int64_t ref_rowid;
    int32_t compress_buf[10240];
    /* must be last element of struct! */
    INSDC_dna_text seq_buf[TableWriterRefSeq_MAX_SEQ_LEN];
};

struct ReferenceSeq {
    BSTNode dad;
    const ReferenceMgr* mgr;
    bool local;
    char id[64];
    char accession[64];
    bool circular;
    union {
        struct {
            /* local file data */
            uint64_t file_size;
            uint64_t line_sz; /* length of line w/ \n */
            uint64_t fasta_offset; /* first base in file */
            const KMMap* map;
            /* cache last call to KMMapReposition */
            uint64_t map_from;
            uint64_t map_to;
        } local;
        struct {
            const RefSeq* o;
            uint64_t usage;
        } refseq;
    } u;
    /* ref table position */
    int64_t start_rowid;
    /* total reference length */
    uint64_t seq_len;
};

static
int CC ReferenceSeq_Cmp(const void *item, const BSTNode *n)
{
    return strcasecmp((const char*)item, ((const ReferenceSeq*)n)->id);
}

static
int CC ReferenceSeq_Sort(const BSTNode *item, const BSTNode *n)
{
    return ReferenceSeq_Cmp(((const ReferenceSeq*)item)->id, n);
}

static
void CC ReferenceSeq_Unused( BSTNode *node, void *data )
{
    ReferenceSeq* n = (ReferenceSeq*)node;
    ReferenceSeq** d = (ReferenceSeq**)data;

    if( !n->local && n->u.refseq.o != NULL ) {
        if( *d == NULL || (*d)->u.refseq.usage > n->u.refseq.usage ) {
            *d = n;
        }
    }
}

struct OpenConfigFile_ctx {
    char const *name;
    KDirectory const *dir;
    KFile const **kfp;
    rc_t rc;
};

static
bool OpenConfigFile(char const server[], char const volume[], void *Ctx)
{
    struct OpenConfigFile_ctx *ctx = Ctx;
    KDirectory const *dir;
    
    if( volume == NULL ) {
        ctx->rc = KDirectoryOpenDirRead(ctx->dir, &dir, false, "%s", server);
    } else {
        ctx->rc = KDirectoryOpenDirRead(ctx->dir, &dir, false, "%s/%s", server, volume);
    }
    if (ctx->rc == 0) {
        ctx->rc = KDirectoryOpenFileRead(dir, ctx->kfp, ctx->name);
        KDirectoryRelease(dir);
        if (ctx->rc == 0) {
            return true;
        }
    }
    return false;
}

static
rc_t FindAndOpenConfigFile(const RefSeqMgr* rmgr, KDirectory const *dir, KFile const **kfp, char const conf[])
{
    rc_t rc = KDirectoryOpenFileRead(dir, kfp, conf);
    
    if(rc) {
        struct OpenConfigFile_ctx ctx;

        ctx.name = conf;
        ctx.dir = dir;
        ctx.kfp = kfp;
        ctx.rc = 0;
        
        rc = RefSeqMgr_ForEachVolume(rmgr, OpenConfigFile, &ctx);
        if (rc == 0 && *kfp == NULL) {
            rc = RC(rcAlign, rcIndex, rcConstructing, rcFile, rcNotFound);
        }
    }
    return rc;
}

static
rc_t ReferenceMgr_Conf(const RefSeqMgr* rmgr, BSTree* tree, KDirectory const *dir, VDBManager const *vmgr, char const conf[])
{
    rc_t rc = 0;
    const KFile* kf = NULL;

    assert(tree != NULL);
    assert(dir != NULL);

    if( conf != NULL &&
        (rc = FindAndOpenConfigFile(rmgr, dir, &kf, conf)) == 0 ) {
        const KMMap* mm;
        uint64_t pos = 0;
        size_t page_sz;
        const char* str;

        if( (rc = KMMapMakeRead(&mm, kf)) == 0 ) {
            do {
                ReferenceSeq* tmp = NULL;
                if( (rc = KMMapReposition(mm, pos, &page_sz)) == 0 &&
                    (rc = KMMapAddrRead(mm, (const void**)&str)) == 0 ) {
                    while( rc == 0 && page_sz > 0 ) {
                        const char* n = memchr(str, '\n', page_sz);
                        const char* r = memchr(str, '\r', page_sz);

                        if( (tmp = calloc(1,  sizeof(*tmp))) == NULL ) {
                            rc = RC(rcAlign, rcIndex, rcConstructing, rcMemory, rcExhausted);
                            break;
                        }
                        if(n == NULL && r == NULL) {
                            n = str + page_sz + 1;
                            page_sz = 0;
                        } else {
                            if( n == NULL || (r != NULL && r < n) ) {
                                n = r;
                            }
                            page_sz -= n - str + 1;
                        }
                        r = str;
                        /* find id: chr6 */
                        while( r < n && !isspace(*r)){ r++; }
                        if( r == n ) {
                            rc = RC(rcAlign, rcIndex, rcConstructing, rcFileFormat, rcInvalid);
                            break;
                        } else if( r - str >= sizeof(tmp->id) ) {
                            rc = RC(rcAlign, rcIndex, rcConstructing, rcBuffer, rcInsufficient);
                            break;
                        } else {
                            memcpy(tmp->id, str, r - str);
                            tmp->id[r - str] = '\0';
                        }
                        /* skip space(s) */
                        while( r < n && isspace(*r)){ r++; }
                        if( r >= n ) {
                            rc = RC(rcAlign, rcIndex, rcConstructing, rcFileFormat, rcInvalid);
                            break;
                        }
                        str = r;
                        /* get accession */
                        while( r < n && !isspace(*r)){ r++; }
                        if( r - str >= sizeof(tmp->accession) ) {
                            rc = RC(rcAlign, rcIndex, rcConstructing, rcBuffer, rcInsufficient);
                            break;
                        } else {
                            memcpy(tmp->accession, str, r - str);
                            tmp->accession[r - str] = '\0';
                        }
                        if( RefSeqMgr_Exists(rmgr, tmp->accession, r - str) == 0 ) {
                            tmp->local = false;
                            if( (rc = BSTreeInsertUnique(tree, &tmp->dad, NULL, ReferenceSeq_Sort)) != 0 ) {
                                break;
                            }
                            ALIGN_DBG("RefSeq translation added '%s' -> '%s'", tmp->id, tmp->accession);
                        } else {
                            /* skips unknown references, later may be picked up as local files */
                            (void)PLOGMSG(klogWarn, (klogWarn, "RefSeq table '$(acc)' for '$(id)' was not found",
                                                               "acc=%s,id=%s", tmp->accession, tmp->id));
                            free(tmp);
                        }
                        tmp = NULL;
                        str = n + 1;
                    }
                }
                free(tmp);
            } while(rc == 0 && page_sz > 0);
            KMMapRelease(mm);
        }
        KFileRelease(kf);
    }
    return rc;
}

LIB_EXPORT rc_t CC ReferenceMgr_Make(const ReferenceMgr** cself, VDatabase* db, const VDBManager* vmgr,
                                     const uint32_t options, const char* conf, const char* path,
                                     uint32_t max_seq_len, size_t cache, uint32_t num_open)
{
    rc_t rc = 0;
    KDirectory* dir = NULL;
    ReferenceMgr* obj = NULL;
    uint32_t wopt = 0;
    
    wopt |= (options & ewrefmgr_co_allREADs) ? ewref_co_SaveRead : 0;
    wopt |= (options & ewrefmgr_co_Coverage) ? ewref_co_Coverage : 0;

    if( max_seq_len == 0 ) {
        max_seq_len = TableWriterRefSeq_MAX_SEQ_LEN;
    }
    if( cself == NULL || (vmgr == NULL && conf != NULL) ) {
        rc = RC(rcAlign, rcIndex, rcConstructing, rcParam, rcNull);
    } else if( (obj = calloc(1, sizeof(*obj) + max_seq_len - sizeof(obj->seq_buf))) == NULL ) {
        rc = RC(rcAlign, rcIndex, rcConstructing, rcMemory, rcExhausted);

    } else if( (rc = KDirectoryNativeDir(&dir)) == 0 ) {
        BSTreeInit(&obj->tree);
        obj->options = options;
        obj->cache = cache;
        obj->num_open_max = num_open;
        obj->max_seq_len = max_seq_len;
        if( (rc = VDBManagerAddRef(vmgr)) == 0 ) {
            obj->vmgr = vmgr;
        }
        if( rc == 0 && db != NULL && (rc = VDatabaseAddRef(db)) == 0 ) {
            obj->db = db;
        }
        if( rc == 0 ) {
            if( path == NULL ) {
                if( (rc = KDirectoryAddRef(dir)) == 0 ) {
                    obj->dir = dir;
                }
            } else {
                rc = KDirectoryOpenDirRead(dir, &obj->dir, false, path);
            }
            if( rc == 0 ) {
                rc = RefSeqMgr_Make(&obj->rmgr, obj->vmgr, 0, cache, num_open);
            }
        }
        if (rc == 0 && (rc = ReferenceMgr_Conf(obj->rmgr, &obj->tree, obj->dir, obj->vmgr, conf)) != 0 ) {
            (void)PLOGERR(klogErr, (klogErr, rc, "failed to open configuration $(file)", "file=%s/%s", path ? path : ".", conf));
        }
    }
    KDirectoryRelease(dir);
    if( rc == 0 ) {
        *cself = obj;
        ALIGN_DBG("conf %s, local path '%s'", conf ? conf : "", path ? path : "");
    } else {
        ReferenceMgr_Release(obj, false, NULL, false);
        ALIGN_DBGERR(rc);
    }
    return rc;
}

static
void CC ReferenceSeq_Whack(BSTNode *n, void *data)
{
    ReferenceSeq* self = (ReferenceSeq*)n;
    if( self->local ) {
        KMMapRelease(self->u.local.map);
    } else {
        RefSeq_Release(self->u.refseq.o);
    }
    free(self);
}

#define ID_CHUNK_SZ 16
typedef struct TChunk_struct {
    struct TChunk_struct* next;
    uint16_t secondary; /* bit mask */
    int64_t id[ID_CHUNK_SZ];
} TChunk;

typedef struct {
    ReferenceSeqCoverage cover;
    TChunk* head;
    TChunk* tail;
    uint8_t tail_qty; /* id count within tail */
    uint32_t id_qty; /* id count overall */
    uint8_t* hilo; /* array of max_seq_len */
} TCover;

static
void ReferenceMgr_TCoverRelease(const TCover* c)
{
    TChunk* h = c->head;
    while( h != NULL ) {
        TChunk* n = h->next;
        free(h);
        h = n;
    }
}
static
int CC int64_cmp( const void* l, const void* r, void *data )
{
    return *(const int64_t*)l - *(const int64_t*)r;
}

static
rc_t ReferenceMgr_ReCover(const ReferenceMgr* cself, uint64_t rows)
{
    rc_t rc = 0;
    int i;
    uint64_t new_rows = 0;
    const TableWriterRefCoverage* cover_writer = NULL;

    TableReaderColumn acols[] =
    {
        {0, "REF_ID", NULL, 0, 0},
        {0, "REF_START", NULL, 0, 0},
        {0, "REF_LEN", NULL, 0, 0},
        {0, "MISMATCH", NULL, 0, 0},
        {0, "REF_OFFSET", NULL, 0, 0},
        {0, "HAS_REF_OFFSET", NULL, 0, 0},
        {0, NULL, NULL, 0, 0}
    };
    const int64_t** al_ref_id = (const int64_t**)&acols[0].base;
    const INSDC_coord_zero** al_ref_start = (const INSDC_coord_zero**)&acols[1].base;
    const INSDC_coord_len** al_ref_len =  (const INSDC_coord_len**)&acols[2].base;
    const uint32_t* mismatch_len =  &acols[3].len;
    const uint32_t* ref_offset_len =  &acols[4].len;
    const int32_t** ref_offset = (const int32_t **)&acols[4].base;
    const uint8_t** has_ref_offset = (const uint8_t **)&acols[5].base;

    /* TODO TO DO TBD change to work properly with mutiple reads in alignment table */

    if( (rc = TableWriterRefCoverage_Make(&cover_writer, cself->db)) == 0 ) {
        rc_t rc1 = 0;
        /* array of ids for sorting and writing to table:
           for head prim_ids , from tail secondry_ids */
        int64_t ref_from = 1, *ids = NULL, rr;
        size_t ids_sz = 0;
        uint64_t ref_qty = rows;

        ALIGN_DBG("sizeof(TCover) %u, sizeof(TChunk) %u", sizeof(TCover), sizeof(TChunk));
        while( rc == 0 && ref_from <= rows ) {
            const char* tbls[] = {"PRIMARY_ALIGNMENT", "SECONDARY_ALIGNMENT"};
            TCover* data = NULL;
            uint8_t* hilo;
            /* allocate mem for up to ref_qty of reference coverage data up to 4Gb */
            do {
                TCover* t = NULL;
                if( (ref_qty * (sizeof(*t) + cself->max_seq_len)) > (4L * 1024 * 1024 * 1024) ||
                    (t = calloc(ref_qty, sizeof(*t) + cself->max_seq_len)) == NULL ) {
                    ref_qty = ref_qty / 3 * 2;
                }
                if( ref_qty < 1 ) {
                    rc = RC(rcAlign, rcTable, rcCommitting, rcMemory, rcExhausted);
                }
                data = t;
                hilo = (uint8_t*)&data[ref_qty];
            } while( rc == 0 && data == NULL );
            /* grep through tables for coverage data */
            ALIGN_DBG("covering REFERENCE rowid range [%ld:%ld]", ref_from, ref_from + ref_qty - 1);
            for(i = 0; rc == 0 && i < sizeof(tbls)/sizeof(tbls[0]); i++ ) {
                const VTable* table = NULL;
                const TableReader* reader = NULL;
                int64_t al_from;
                uint64_t al_qty;

                if( (rc = VDatabaseOpenTableRead(cself->db, &table, tbls[i])) == 0 &&
                    (rc = TableReader_Make(&reader, table, acols, cself->cache)) == 0 &&
                    (rc = TableReader_IdRange(reader, &al_from, &al_qty)) == 0 ) {
                    int64_t al_rowid;
                    for(al_rowid = al_from; rc == 0 && al_rowid <= al_qty; al_rowid++) {
                        int64_t ref_r, al_ref_id_end;
                        uint64_t j, seq_start = 0, refseq_start, refseq_len;
                        if( (rc = TableReader_ReadRow(reader, al_rowid)) != 0 ) {
                            break;
                        }
                        /* alignment can run across multiple reference chunks */
                        al_ref_id_end = cself->max_seq_len - **al_ref_start;
                        if( al_ref_id_end < **al_ref_len ) {
                            al_ref_id_end = (**al_ref_len - al_ref_id_end) / cself->max_seq_len + 1;
                        } else {
                            al_ref_id_end = 1;
                        }
                        al_ref_id_end += **al_ref_id;
                        refseq_start = **al_ref_start;
                        ALIGN_DBG("al row %li has REF_ID [%ld,%ld], REF_START: %u, REF_LEN %u",
                                    al_rowid, **al_ref_id, al_ref_id_end - 1, **al_ref_start, **al_ref_len);
                        for(ref_r = **al_ref_id; ref_r < al_ref_id_end; ref_r++) {
                            refseq_len = cself->max_seq_len - refseq_start;
                            if( seq_start + refseq_len > **al_ref_len ) {
                                refseq_len = **al_ref_len - seq_start;
                            }
                            ALIGN_DBG("covered ref_id %ld [%ld,%ld]", ref_r, refseq_start, refseq_start + refseq_len);
                            if( ref_r >= ref_from && ref_r < ref_from + ref_qty ) {
                                uint64_t k = ref_r - ref_from;
                                ALIGN_DBG("%ld is a match for %ld[%lu]", ref_r, ref_from + k, k);
                                if( data[k].tail_qty == ID_CHUNK_SZ || data[k].head == NULL ) {
                                    TChunk* x = malloc(sizeof(*(data[k].tail)));
                                    while( x == NULL ) {
                                        /* release last ref cover_writer record and retry */
                                        ReferenceMgr_TCoverRelease(&data[--ref_qty]);
                                        ALIGN_DBG("downsize covering REFERENCE rowid range [%ld:%ld] from %s",
                                                  ref_from, ref_from + ref_qty - 1, tbls[i]);
                                        if( ref_qty < 1 ) {
                                            rc = RC(rcAlign, rcTable, rcCommitting, rcMemory, rcExhausted);
                                            break;
                                        } else if( ref_r >= ref_from + ref_qty ) {
                                            break;
                                        }
                                    }
                                    if( rc != 0 ) {
                                        break;
                                    }
                                    if( data[k].head == NULL ) {
                                        data[k].head = x;
                                    } else {
                                        data[k].tail->next = x;
                                    }
                                    x->secondary = 0;
                                    data[k].tail = x;
                                    data[k].tail->next = NULL;
                                    data[k].tail_qty = 0;
                                    data[k].hilo = &hilo[cself->max_seq_len * k];
                                }
                                data[k].tail->id[data[k].tail_qty] = al_rowid;
                                if( i > 0 ) {
                                    data[k].tail->secondary |= 1 << data[k].tail_qty;
                                }
                                data[k].tail_qty++;
                                data[k].id_qty++;
                                if( ref_r == **al_ref_id ) {
                                    /* write those to 1st chunk only */
                                    
                                    unsigned left_soft_clip = 0;
                                    
                                    if (*ref_offset_len > 0 && (*has_ref_offset)[0] != 0 && (*ref_offset)[0] < 0) {
                                        left_soft_clip = 1;
                                    }
                                    data[k].cover.mismatches += *mismatch_len;
                                    data[k].cover.indels += *ref_offset_len - left_soft_clip;
                                }
                                for(j = refseq_start; j < refseq_start + refseq_len; j++) {
                                    if( data[k].hilo[j] < UINT8_MAX ) {
                                        data[k].hilo[j]++;
                                    }
                                }
                            }
                            seq_start += refseq_len;
                            refseq_start = 0;
                        }
                        rc = rc ? rc : Quitting();
                    }
                }
                VTableRelease(table);
                TableReader_Whack(reader);
            }
            /* prep and write coverage data */
            for(rr = 0; rc == 0 && rr < ref_qty; rr++) {
                uint32_t i;
                TChunk* x;
                ReferenceSeqCoverage* c = &data[rr].cover;

                while( ids_sz < data[rr].id_qty ) {
                    int64_t* n = realloc(ids, data[rr].id_qty * sizeof(*ids));
                    if( n == NULL ) {
                        ReferenceMgr_TCoverRelease(&data[--ref_qty]);
                        ALIGN_DBG("downsize covering REFERENCE rowid range [%ld:%ld]", ref_from, ref_from + ref_qty - 1);
                        if( ref_qty < 1 ) {
                            rc = RC(rcAlign, rcTable, rcCommitting, rcMemory, rcExhausted);
                            break;
                        } else if( rr >= ref_qty ) {
                            break;
                        }
                    } else {
                        ids = n;
                        ids_sz = data[rr].id_qty;
                    }
                }
                if( rc != 0 || rr >= ref_qty ) {
                    break;
                }
                ALIGN_DBGF(("ref rowid %ld:", ref_from + rr));
                x = data[rr].head;
                while( x != NULL ) {
                    uint8_t q = data[rr].tail == x ? data[rr].tail_qty : ID_CHUNK_SZ;
                    for(i = 0; i < q; i++) {
                        ALIGN_DBGF((" %lu[%c]", x->id[i], (x->secondary & i) ? 's' : 'p'));
                        if( x->secondary & i ) {
                            ids[ids_sz - ++c->secondary_ids.elements] = x->id[i];
                        } else {
                            ids[c->primary_ids.elements++] = x->id[i];
                        }
                    }
                    x = x->next;
                }
                c->primary_ids.buffer = ids;
                c->secondary_ids.buffer = &ids[ids_sz - c->secondary_ids.elements];
                ALIGN_DBGF((" ids 0x%p:0x%p,", ids, ids + ids_sz));
                ALIGN_DBGF((" prm 0x%p q=%lu,", c->primary_ids.buffer, c->primary_ids.elements));
                ALIGN_DBGF((" 2nd 0x%p q=%lu\n", c->secondary_ids.buffer, c->secondary_ids.elements));
                ksort((void*)(c->primary_ids.buffer), c->primary_ids.elements, sizeof(*ids), int64_cmp, NULL);
                ksort((void*)(c->secondary_ids.buffer), c->secondary_ids.elements, sizeof(*ids), int64_cmp, NULL);
                if( data[rr].hilo != NULL ) {
                    memset(&c->low, 0xFF, sizeof(c->low));
                    for(i = 0; i < cself->max_seq_len; i++) {
                        if( c->high < data[rr].hilo[i] ) {
                            c->high = data[rr].hilo[i];
                        }
                        if( c->low > data[rr].hilo[i] ) {
                            c->low = data[rr].hilo[i];
                        }
                    }
                }
                rc = TableWriterRefCoverage_Write(cover_writer, ref_from + rr, c);
                ReferenceMgr_TCoverRelease(&data[rr]);
            }
            free(data);
            ref_from += ref_qty;
            ref_qty = rows - ref_from + 1;
        }
        free(ids);
        rc1 = TableWriterRefCoverage_Whack(cover_writer, rc == 0, &new_rows);
        rc = rc ? rc : rc1;
        if( rc == 0 && rows != new_rows ) {
            rc = RC(rcAlign, rcTable, rcCommitting, rcData, rcInconsistent);
        }
    }
    return rc;
}

LIB_EXPORT rc_t CC ReferenceMgr_Release(const ReferenceMgr* cself, bool commit, uint64_t* rows, bool build_coverage)
{
    rc_t rc = 0;
    if( cself != NULL ) {
        uint64_t rr, *r = rows ? rows : &rr;
        ReferenceMgr* self = (ReferenceMgr*)cself;

        rc = TableWriterRef_Whack(self->writer, commit, r);
        BSTreeWhack(&self->tree, ReferenceSeq_Whack, NULL);
        KDirectoryRelease(self->dir);
        if( rc == 0 && build_coverage ) {
            rc = ReferenceMgr_ReCover(cself, *r);
        }
        VDatabaseRelease(self->db);
        VDBManagerRelease(self->vmgr);
        RefSeqMgr_Release(self->rmgr);
        free(self);
    }
    return rc;
}

static
rc_t ReferenceSeq_Local(ReferenceSeq** self, BSTree* tree, const char* id, const KFile* kf)
{
    rc_t rc = 0;
    ReferenceSeq* obj;
    size_t page_sz;
    
    assert(self != NULL);
    assert(id != NULL);
    assert(kf != NULL);
        
    *self = NULL;
    if( (obj = calloc(1, sizeof(*obj))) == NULL ) {
        rc = RC(rcAlign, rcFile, rcConstructing, rcMemory, rcExhausted);
    } else if( ((obj->local = true)) &&
               (rc = KFileSize(kf, &obj->u.local.file_size)) == 0 &&
               (rc = KMMapMakeRead(&obj->u.local.map, kf)) == 0 &&
               (rc = KMMapReposition(obj->u.local.map, 0, &page_sz)) == 0 ) {
        const char* addr;

        obj->u.local.map_to = page_sz;
        /* make sure we have 2 lines */
        if( (rc = KMMapAddrRead(obj->u.local.map, (const void**)&addr)) == 0 ) {
            const char* sp = memchr(addr, ' ', page_sz);
            const char* nl1 = memchr(addr, '\n', page_sz);
            const char* nl2 = memchr(nl1 + 1, '\n', page_sz - (nl1 - addr));
            sp = sp ? sp : nl1;
            if( addr[0] != '>' || nl1 == NULL || nl2 == NULL || sp-- > nl1 ) {
                rc = RC(rcAlign, rcFile, rcConstructing, rcData, rcUnrecognized);
            } else if( strlen(id) >= sizeof(obj->id) || (sp - addr) >= sizeof(obj->accession) ) {
                rc = RC(rcAlign, rcFile, rcConstructing, rcBuffer, rcInsufficient);
            } else {
                strcpy(obj->id, id);
                memcpy(obj->accession, &addr[1], sp - addr);
                obj->accession[sp - addr] = '\0';
                obj->u.local.line_sz = nl2 - nl1 - 1;
                obj->u.local.fasta_offset = (nl1 - addr) + 1;
                obj->seq_len = obj->u.local.file_size - obj->u.local.fasta_offset;
                obj->seq_len -= obj->seq_len / obj->u.local.line_sz; /* remove \n qty */
                rc = BSTreeInsertUnique(tree, &obj->dad, NULL, ReferenceSeq_Sort);
            }
        }
    }
    if( rc == 0 ) {
        *self = obj;
        ALIGN_DBG("RefSeq local fasta added '%s' -> '%s'", obj->id, obj->accession);
    } else {
        ReferenceSeq_Whack(&obj->dad, NULL);
        ALIGN_DBGERR(rc);
    }
    return rc;
}

static
rc_t ReferenceSeq_ReOffset(const ReferenceSeq* self, int64_t* offset)
{
    /* see TableReaderRefSeq_Read, same logic */
    if( !self->circular && (*offset < 0 || *offset >= self->seq_len) ) {
        return RC(rcAlign, rcType, rcReading, rcOffset, rcOutofrange);
    } else if( *offset < 0 ) {
        *offset = self->seq_len - ((-(*offset)) % self->seq_len);
    } else if( self->circular && *offset > self->seq_len ) {
        *offset %= self->seq_len;
    }
    return 0;
}

static
rc_t ReferenceSeq_ReadDirect(ReferenceSeq* self, int64_t offset, INSDC_coord_len len, bool read_curcular,
                             INSDC_dna_text* buffer, INSDC_coord_len* written)
{
    rc_t rc = 0;

    *written = 0;
    if( (rc = ReferenceSeq_ReOffset(self, &offset)) != 0 ) {
    } else if( self->local ) {
        /* translate offset into file dimensions */
        offset += offset / self->u.local.line_sz; /* add \n on each line */
        offset += self->u.local.fasta_offset; /* add defline */
        if( offset >= self->u.local.file_size ) {
            return RC(rcAlign, rcFile, rcReading, rcOffset, rcOutofrange);
        } else {
            size_t page;
            const char* str;
            do {
                if( offset < self->u.local.map_from || self->u.local.map_to < offset ) {
                    if( (rc = KMMapReposition(self->u.local.map, offset, &page)) == 0 ) {
                        self->u.local.map_from = offset;
                        self->u.local.map_to = offset + page;
                    }
                }
                if( rc == 0 && (rc = KMMapAddrRead(self->u.local.map, (const void**)&str)) == 0 ) {
                    offset += page;
                    while( page > 0 && len > 0 ) {
                        char* nl = memchr(str, '\n', page);
                        size_t q = (nl == NULL) ? page : nl - str;
                        if( q > len ) {
                            q = len;
                        }
                        memcpy(&buffer[*written], str, q);
                        *written = *written + q;
                        len -= q;
                        if( nl != NULL ) {
                            q++;
                        }
                        page -= q;
                        str += q;
                    }
                }
                if( read_curcular && offset >= self->u.local.file_size ) {
                    offset = self->u.local.fasta_offset;
                }
            } while( rc == 0 && len > 0 && offset < self->u.local.file_size );
        }
    } else {
        /* we need to trim len to actual length of seq */
        if( !read_curcular && (offset + len) >= self->seq_len ) {
            len = self->seq_len - offset;
        }
        if( rc == 0 && (rc = RefSeq_Read(self->u.refseq.o, offset, len, buffer, written)) == 0 ) {
            self->u.refseq.usage = ++((ReferenceMgr*)self->mgr)->usage;
        }
    }
    ALIGN_DBGERR(rc);
    return rc;
}

static
rc_t CC ReferenceMgr_Find(const ReferenceMgr* cself, const char* id, ReferenceSeq** seq, const KFile** kf)
{
    *seq = (ReferenceSeq*)BSTreeFind(&cself->tree, id, ReferenceSeq_Cmp);
    if( *seq == NULL ) {
        /* try local file */
        return KDirectoryOpenFileRead(cself->dir, kf, "%s.fasta", id);
    }
    *kf = NULL;
    return 0;
}

LIB_EXPORT rc_t CC ReferenceMgr_GetSeq(const ReferenceMgr* cself, const ReferenceSeq** seq, const char* id)
{
    rc_t rc = 0;
    ReferenceSeq* obj;
    
    if( cself == NULL || seq == NULL || id == NULL ) {
        rc = RC(rcAlign, rcFile, rcConstructing, rcParam, rcNull);
    } else {
        ReferenceMgr* mgr = (ReferenceMgr*)cself;
        const KFile* kf;
        
        *seq = NULL;
        rc = ReferenceMgr_Find(cself, id, &obj, &kf);
        if( rc == 0 && obj == NULL ) {
            /* it is local file */
            rc = ReferenceSeq_Local(&obj, &mgr->tree, id, kf);
            KFileRelease(kf);
        }
        if( rc == 0 && !obj->local && obj->u.refseq.o == NULL ) {
            if( cself->num_open_max > 0 && cself->num_open >= cself->num_open_max ) {
                ReferenceSeq* old = NULL;
                BSTreeForEach(&cself->tree, false, ReferenceSeq_Unused, &old);
                if( old != NULL ) {
                    RefSeq_Release(old->u.refseq.o);
                    old->u.refseq.o = NULL;
                    mgr->num_open--;
                }
            }
            rc = RefSeqMgr_GetSeq(cself->rmgr, &obj->u.refseq.o, obj->accession, strlen(obj->accession));
            if( rc == 0 &&
                (rc = RefSeq_Circular(obj->u.refseq.o, &obj->circular)) == 0 &&
                (rc = RefSeq_SeqLength(obj->u.refseq.o, &obj->seq_len)) == 0 ) {
                mgr->num_open++;
            }
        }
        if( rc == 0 && obj->start_rowid == 0 ) {
            /* append to the whole thing to REFERENCE table since we encounter it for the 1st time */
            TableWriterRefData data;
            INSDC_coord_len len = 0;
            int64_t offset = 0;
            obj->mgr = cself;
            obj->start_rowid = mgr->ref_rowid + 1;
            data.name.buffer = obj->id;
            data.name.elements = strlen(obj->id);
            data.read.buffer = mgr->seq_buf;
            data.seq_id.buffer = obj->accession;
            data.seq_id.elements = strlen(obj->accession);
            data.force_READ_write = obj->local || (cself->options & ewrefmgr_co_allREADs);
            data.circular = obj->circular;
            if (cself->writer == NULL) {
                uint32_t wopt = 0;

                wopt |= (mgr->options & ewrefmgr_co_allREADs) ? ewref_co_SaveRead : 0;
                wopt |= (mgr->options & ewrefmgr_co_Coverage) ? ewref_co_Coverage : 0;
                if( (rc = TableWriterRef_Make(&mgr->writer, mgr->db, wopt)) == 0 ) {
                    TableWriterData mlen;
                    mlen.buffer = &mgr->max_seq_len;
                    mlen.elements = 1;
                    rc = TableWriterRef_WriteDefaultData(mgr->writer, ewrefseq_cn_MAX_SEQ_LEN, &mlen);
                }
            }
            if (rc == 0) {
                do {
                    if( (rc = ReferenceSeq_ReadDirect(obj, offset, mgr->max_seq_len, false,
                                                      mgr->seq_buf, &len)) == 0 && len > 0 ) {
                        data.read.elements = len;
                        rc = TableWriterRef_Write(cself->writer, &data, NULL);
                        offset += len;
                        mgr->ref_rowid++;
                    }
                } while( rc == 0 && len > 0 && offset < obj->seq_len );
            }
        }
    }
    if( rc == 0 ) {
        *seq = obj;
    } else {
        ALIGN_DBGERR(rc);
    }
    return rc;
}

LIB_EXPORT rc_t CC ReferenceMgr_Verify(const ReferenceMgr* cself, const char* id, uint64_t seq_len, const uint8_t* md5)
{
    rc_t rc = 0;
    ReferenceSeq* rseq;
    const KFile* kf;
    
    if( cself == NULL || id == NULL ) {
        rc = RC(rcAlign, rcFile, rcConstructing, rcParam, rcNull);
    } else if( (rc = ReferenceMgr_Find(cself, id, &rseq, &kf)) == 0 ) {
        if( rseq == NULL ) {
            uint64_t size = 0;
            if( (rc = KFileSize(kf, &size)) == 0 && size == 0 ) {
                rc = RC(rcAlign, rcTable, rcValidating, rcSize, rcEmpty);
            }
            KFileRelease(kf);
        } else if( rseq->local ) {
            if( rseq->seq_len != seq_len ) {
                rc = RC(rcAlign, rcFile, rcValidating, rcSize, rcUnequal);
                ALIGN_DBGERRP("%s->%s SEQ_LEN verification", rc, id, rseq->accession);
            }
        } else {
            const RefSeq* tmp = rseq->u.refseq.o;
            uint64_t o_len;
            const uint8_t* o_md5;

            if( tmp == NULL ) {
                if( (rc = RefSeqMgr_GetSeq(cself->rmgr, &tmp, rseq->accession, strlen(rseq->accession))) != 0 ||
                    (rc = RefSeq_SeqLength(tmp, &o_len)) != 0 ) {
                    ALIGN_DBGERRP("%s->%s verification", rc, id, rseq->accession);
                }
            } else {
                o_len = rseq->seq_len;
            }
            if( rc == 0 ) {
                if( seq_len != 0 && o_len != seq_len ) {
                    rc = RC(rcAlign, rcTable, rcValidating, rcSize, rcUnequal);
                    ALIGN_DBGERRP("%s->%s SEQ_LEN verification", rc, id, rseq->accession);
                } else {
                    ALIGN_DBG("%s->%s SEQ_LEN verification ok", id, rseq->accession);
                }
            }
            if( rc == 0 && (rc = RefSeq_MD5(tmp, &o_md5)) == 0 ) {
                if( md5 != NULL && o_md5 != NULL && memcmp(md5, o_md5, 16) != 0 ) {
                    rc = RC(rcAlign, rcTable, rcValidating, rcChecksum, rcUnequal);
                    ALIGN_DBGERRP("%s->%s MD5 verification", rc, id, rseq->accession);
                } else {
                    ALIGN_DBG("%s->%s MD5 verification ok", id, rseq->accession);
                }
            }
            if( tmp != rseq->u.refseq.o ) {
                RefSeq_Release(tmp);
            }
        }
    }
    if( rc == 0 ) {
        ALIGN_DBG("%s verification ok", id);
    } else {
        ALIGN_DBGERRP("%s verification", rc, id);
    }
    return rc;
}

LIB_EXPORT rc_t CC ReferenceMgr_Compress(const ReferenceMgr* cself, uint32_t options,
                                         const char* id, int64_t offset,
                                         const char* seq, INSDC_coord_len seq_len,
                                         const void* cigar, uint32_t cigar_len,
                                         TableWriterAlgnData* data)
{
    rc_t rc = 0;
    const ReferenceSeq* refseq;

    if( cself == NULL || id == NULL ) {
        rc = RC(rcAlign, rcFile, rcProcessing, rcParam, rcNull);
    } else if( (rc = ReferenceMgr_GetSeq(cself, &refseq, id)) == 0 ) {
        rc = ReferenceSeq_Compress(refseq, options, offset, seq, seq_len, cigar, cigar_len, data);
        ReferenceSeq_Release(refseq);
    }
    ALIGN_DBGERR(rc);
    return rc;
}

LIB_EXPORT rc_t CC ReferenceSeq_Compress(const ReferenceSeq* cself, uint32_t options, int64_t offset,
                                         const char* seq, INSDC_coord_len seq_len,
                                         const void* cigar, uint32_t cigar_len,
                                         TableWriterAlgnData* data)
{
    rc_t rc = 0;

    ALIGN_DBG("align '%.*s'[%u] to '%s' at %li using '%.*s'[%u]",
        (int)seq_len, seq, seq_len, cself ? cself->id : NULL, offset, 
          (options & ewrefmgr_cmp_Binary) ? 3 : (int)cigar_len, (options & ewrefmgr_cmp_Binary) ? "bin" : cigar, cigar_len);

    if( cself == NULL || seq == NULL || cigar == NULL || cigar_len == 0 || data == NULL ) {
        rc = RC(rcAlign, rcFile, rcProcessing, rcParam, rcInvalid);
    } else if( seq_len > sizeof(cself->mgr->compress_buf) / sizeof(cself->mgr->compress_buf[0]) ) {
        rc = RC(rcAlign, rcFile, rcProcessing, rcBuffer, rcInsufficient);
    } else if( (rc = ReferenceSeq_ReOffset(cself, &offset)) == 0 ) {
        INSDC_coord_len seq_pos = 0;
        uint32_t i, last_match = 0, op_len;
        unsigned char op;
        INSDC_coord_zero* read_start = &((INSDC_coord_zero*)(data->read_start.buffer))[data->ploidy];
        INSDC_coord_len* read_len = &((INSDC_coord_len*)(data->read_len.buffer))[data->ploidy];
        INSDC_coord_len rl = 0, *ref_len = &((INSDC_coord_len*)(data->ref_len.buffer))[data->ploidy];
        bool* has_ref_offset;
        uint64_t i_ref_offset_elements;
        int32_t* ref_offset, *cb = (int32_t*)(cself->mgr->compress_buf);

        if( data->ploidy == 0 ) {
            data->has_ref_offset.elements = seq_len;
            data->ref_offset.elements = 0;
            *read_start = 0;
            i_ref_offset_elements = 0;
        } else {
            data->has_ref_offset.elements += seq_len;
            *read_start = read_start[-1] + read_len[-1];
            i_ref_offset_elements = data->ref_offset.elements;
        }
        *read_len = seq_len;
        *ref_len = 0;

#if _DEBUGGING
        ALIGN_DBG("real offset %li ref is circular? %s", offset, cself->circular ? "true" : "false");
        if( options & ewrefmgr_cmp_Binary ) {
            ALIGN_DBGF(("%s:%u: bin cigar:", __func__, __LINE__));
            for(i = 0; i < cigar_len; i++) {
                const uint32_t* c = cigar;
                op = c[i] & 0x0F;
                op_len = c[i] >> 4;
                ALIGN_DBGF(("%u%c", op_len, "MIDNSHP=XB"[op]));
            }
            ALIGN_DBGF(("\n"));
        }
#endif
        memset(cb, 0, sizeof(*cb) * seq_len);
        for(i = 0; rc == 0 && i < cigar_len; i++) {
            if( options & ewrefmgr_cmp_Binary ) {
                const uint32_t* c = cigar;
                op = c[i] & 0x0F;
                op_len = c[i] >> 4;
            } else {
                const unsigned char* c = cigar;
                op_len = 0;
                while( c[i] != '\0' && isdigit(c[i]) ) {
                    op_len *= 10;
                    op_len += c[i++] - '0';
                }
                /* avoid intersecting with 4-bit binary above */
                op = c[i] <= 0x0F ? 0xFF : c[i];
            }
            switch(op) {
                case 'M':
                case 0:
                case '=':
                case 7:
                case 'X':
                case 8:
                    seq_pos += op_len;
                    rl += op_len;
                    *ref_len = rl;
                    last_match = seq_pos;
                    break;
                case 'I':
                case 1:
                case 'S':
                case 4:
                    seq_pos += op_len;
                    cb[last_match] += -op_len;
                    break;
                case 'B':
                case 9:
                    /* Complete Genomics CIGAR style specific:
                       overlap between consecutive reads
                       ex: sequence 6 bases: ACACTG, reference 2 bases: ACTG,
                       cigar will be: 2M2B2M
                       no need to move sequence position
                    */
                    cb[last_match] -= op_len;
                    if( rl < op_len ) {
                        /* step back beyond ref start */
                        offset -= op_len - rl;
                        rl = 0;
                    } else {
                        rl -= op_len;
                    }
                    break;
                case 'D':
                case 2:
                case 'N':
                case 3:
                    cb[last_match] += op_len;
                    rl += op_len;
                    break;
                case 'P':
                case 6:
                    continue;
                case 'H':
                case 5:
                default:
                    rc = RC(rcAlign, rcFile, rcProcessing, rcData, rcUnrecognized);
            }
        }
        if( rc == 0 ) {
            /* do not write leading deletes just move reference offset */
            if( cb[0] > 0 ) {
                offset += cb[0];
                *ref_len -= cb[0];
                cb[0] = 0;
            }
#if _DEBUGGING
            ALIGN_DBGF(("%s:%u: last_match %u, offsets:", __func__, __LINE__, last_match));
            for(i = 0; i < seq_len; i++) {
                ALIGN_DBGF(("%i%c", cb[i], ((i + 1) % 5) ? ',' : '~'));
            }
            ALIGN_DBGF(("\n"));
#endif
            has_ref_offset = &((bool*)(data->has_ref_offset.buffer))[*read_start];
            ref_offset = (int32_t*)(data->ref_offset.buffer);
            /* do not write trailing indels cause they are simple mismatches */
            for(i = 0; i < last_match; i++) {
                has_ref_offset[i] = cb[i] != 0;
                if( has_ref_offset[i] ) {
                    ref_offset[data->ref_offset.elements++] = cb[i];
                }
            }
            memset(&has_ref_offset[last_match], 0, seq_len - last_match);

#if _DEBUGGING
            ALIGN_DBG("aligned at %li ref_len %u, num offsets %lu, seq_pos %u", offset, *ref_len, data->ref_offset.elements, seq_pos);
            ALIGN_DBGF(("%s:%u: HAS_REF_OFFSET: ", __func__, __LINE__));
            for(i = 0; i < seq_len; i++) {
                ALIGN_DBGF(("%c", (has_ref_offset[i] != 0) ? '1' : '0'));
            }
            ALIGN_DBGF((", REF_OFFSET:"));
            for(i = i_ref_offset_elements; i < data->ref_offset.elements; i++) {
                ALIGN_DBGF((" %i,", ref_offset[i]));
            }
            ALIGN_DBGF(("[%u]\n", data->ref_offset.elements - i_ref_offset_elements));
#endif
            assert(seq_len == seq_pos);
        }
#if _DEBUGGING
        if( rc == 0 ) {
#else
        if( rc == 0 && !(options & ewrefmgr_cmp_NoMismatch) ) {
#endif
            bool* has_mismatch = &((bool*)(data->has_mismatch.buffer))[*read_start];
            INSDC_dna_text* mismatch = (INSDC_dna_text*)(data->mismatch.buffer);
            INSDC_dna_text ref_buf[20480];

            if( *ref_len > sizeof(ref_buf) ) {
                rc = RC(rcAlign, rcFile, rcProcessing, rcBuffer, rcInsufficient);
            } else if( (rc = ReferenceSeq_ReadDirect((ReferenceSeq*)cself, offset, *ref_len,
                                                                      true, ref_buf, &rl)) == 0 ) {
                int64_t ref_pos = 0;
                uint64_t ref_offset_pos = i_ref_offset_elements;
                uint64_t i_mismatch_elements;

                ALIGN_DBG("ref %.*s[%u]", rl, ref_buf, rl);
                if( data->ploidy == 0 ) {
                    data->has_mismatch.elements = 0;
                    data->mismatch.elements = 0;
                }
                i_mismatch_elements = data->mismatch.elements;
                data->has_mismatch.elements += seq_len;

                ALIGN_DBGF(("%s:%u: HAS_MISMATCH: ", __func__, __LINE__));
                for(seq_pos = 0; seq_pos < seq_len; seq_pos++, ref_pos++) {
                    if( has_ref_offset[seq_pos] ) {
                        ref_pos += ref_offset[ref_offset_pos++];
                    }
                    if( (ref_pos < 0) || (ref_pos >= rl) || 
                        ((toupper(ref_buf[ref_pos]) != toupper(seq[seq_pos])) && (seq[seq_pos] != '=')) ) {
                        has_mismatch[seq_pos] = true;
                        mismatch[data->mismatch.elements++] = seq[seq_pos];
                    } else {
                        has_mismatch[seq_pos] = false;
                    }
                    ALIGN_DBGF(("%c", has_mismatch[seq_pos] ? '1' : '0'));
                }
                ALIGN_DBGF((", MISMATCH: '%.*s'[%u]\n", (int)(data->mismatch.elements - i_mismatch_elements),
                    &mismatch[i_mismatch_elements], data->mismatch.elements - i_mismatch_elements));
            }
        }
        if( rc == 0 ) {
            if( data->ref_1st_row_id.buffer ) {
                ((INSDC_coord_len*)(data->ref_1st_row_id.buffer))[data->ploidy] = cself->start_rowid;
                data->ref_1st_row_id.elements = data->ploidy + 1;
                ALIGN_DBG("1st ROW_ID %u", ((INSDC_coord_len*)(data->ref_1st_row_id.buffer))[data->ploidy]);
            }

            if( data->ref_id.buffer ) {
                ((int64_t*)(data->ref_id.buffer))[data->ploidy] = cself->start_rowid + offset / cself->mgr->max_seq_len;
                data->ref_id.elements = data->ploidy + 1;
                ALIGN_DBG("REF_ID: %li", ((int64_t*)(data->ref_id.buffer))[data->ploidy]);
            }
            if( data->ref_start.buffer ) {
                ((INSDC_coord_zero*)(data->ref_start.buffer))[data->ploidy] = offset % cself->mgr->max_seq_len;
                data->ref_start.elements = data->ploidy + 1;
                ALIGN_DBG("REF_START %i", ((INSDC_coord_zero*)(data->ref_start.buffer))[data->ploidy]);
            }
            if( data->global_ref_start.buffer ) {
                ((uint64_t*)(data->global_ref_start.buffer))[data->ploidy] = (cself->start_rowid - 1) * cself->mgr->max_seq_len + offset;
                data->global_ref_start.elements = data->ploidy + 1;
                ALIGN_DBG("GLOBAL_REF_START %lu", ((uint64_t*)(data->global_ref_start.buffer))[data->ploidy]);
            }
            if( data->effective_offset.buffer ) {
                ((int64_t*)(data->effective_offset.buffer))[data->ploidy] = offset;
                data->effective_offset.elements = data->ploidy + 1;
                ALIGN_DBG("OFFSET %li", ((int64_t*)(data->effective_offset.buffer))[data->ploidy]);
            }
            data->ploidy++;
            data->read_start.elements = data->ploidy;
            data->read_len.elements = data->ploidy;
            data->ref_len.elements = data->ploidy;
        }
    }
    ALIGN_DBGERR(rc);
    return rc;
}

LIB_EXPORT rc_t CC ReferenceSeq_Read(const ReferenceSeq* cself, int64_t offset, INSDC_coord_len len,
                                     INSDC_dna_text* buffer, INSDC_coord_len* ref_len)
{
    rc_t rc = 0;

    if( cself == NULL || buffer == NULL || ref_len == NULL ) {
        rc = RC(rcAlign, rcFile, rcReading, rcParam, rcNull);
    } else {
        rc = ReferenceSeq_ReadDirect((ReferenceSeq*)cself, offset, len, true, buffer, ref_len);
    }
    ALIGN_DBGERR(rc);
    return rc;
}

LIB_EXPORT rc_t CC ReferenceSeq_Get1stRow(const ReferenceSeq* cself, int64_t* row_id)
{
    rc_t rc = 0;

    if( cself == NULL || row_id == NULL ) {
        rc = RC(rcAlign, rcFile, rcReading, rcParam, rcNull);
    } else {
        *row_id = cself->start_rowid;
    }
    return rc;
}

LIB_EXPORT rc_t CC ReferenceSeq_AddCoverage(const ReferenceSeq* cself, int64_t offset, const ReferenceSeqCoverage* data)
{
    rc_t rc = 0;

    if( cself == NULL || data == NULL) {
        rc = RC(rcAlign, rcFile, rcReading, rcParam, rcNull);
    } else if( !(cself->mgr->options & ewrefmgr_co_Coverage) ) {
        rc = RC( rcAlign, rcType, rcWriting, rcData, rcUnexpected);
        ALIGN_DBGERRP("coverage %s", rc, "data");
    } else if( (rc = ReferenceSeq_ReOffset(cself, &offset)) == 0 ) {
        rc = TableWriterRef_WriteCoverage(cself->mgr->writer, cself->start_rowid, offset, data);
    }
    ALIGN_DBGERR(rc);
    return rc;
}

LIB_EXPORT rc_t CC ReferenceSeq_Release(const ReferenceSeq *cself)
{
    return 0;
}
