/************************************************************************
* File name: tpconfig.c                                                 *
* Written by: C. Scott Ananian modified by Bruce Kall                   *
* Now Supported by: Bruce Kall (kall@compass.com)                       *
* Language: C                                                           *
* Purpose: This programs configures synaptics touchpads and is compati- *
*   ble with version 3.2+ firmware supporting a single mode byte.       *
************************************************************************/


/*$Log: tpconfig.c,v $
 *Revision 2.13  2002/02/22 20:33:11  bruce
 *Aded for Mode 3 and Mode 4 options.
 *
 *Revision 2.12  2000/11/15 15:37:50  bruce
 *Added back in copyright call().
 *
 *Revision 2.11  2000/11/01 19:56:38  cph
 *Change call to ioctl AUX_GRAB to handle case where it returns EAGAIN.
 *
 *Revision 2.10  2000/10/31 19:29:45  cph
 *Unpatched kernel signals ENOTTY for ioctl.
 *
 *Revision 2.9  2000/10/31 18:57:53  cph
 *Use symbolic value for DEBUG_LEVEL.
 *
 *Revision 2.8  2000/10/31 18:38:12  cph
 *Implement --sleep option to control sleep mode.
 *
 *Revision 2.7  2000/10/31 18:03:22  cph
 *Move copyright into --help screen.  Move variable declarations from .h
 *file to .c file.
 *
 *Revision 2.6  2000/10/31 15:19:49  cph
 *Add support for Linux kernel patch to grab the mouse device.
 *
 *Revision 2.5  2000/10/23 16:28:14  bruce
 *Fixed tcflush call.
 *
 *Revision 2.4  2000/09/29 20:17:39  bruce
 *Changed ALPS_GLIDEPOINT to ALPS_GLIDEPAD.
 *
 *Revision 2.3  2000/09/28 13:37:57  bruce
 *Exatracted synaptics code into synaptics.c.  Added support for
 *ALPS Glidepad/Stickpointer
 *
 *Revision 2.2  1999/02/16 17:52:56  bruce
 *Changed check for 2.2.1 kernel in is_Synaptics().
 *Added to time-out gracefully.
 *
 *Revision 2.1  1999/02/13 22:29:52  bruce
 *Modified to work with newer 2.2.x kernel.
 *
 *Revision 2.0  1998/11/27 14:16:50  bruce
 *Changed to work with modern Synaptic touchpads which contain
 *newer firmware (that have a single mode byte).  Also added
 *detection of newer touchpad info ('Super' 'Ultrathin',
 *'Widepad','Stamppad','Submini','Multiswich').
 *
 *Revision 0.1  1998/07/08 15:23:33  bruce
 *Initial revision by C. Scott Ananian
 **/

static char rcsid[]="$Id: tpconfig.c,v 2.13 2002/02/22 20:33:11 bruce Exp $";



/* tpconfig --- Command-line TouchPad configuration utility.
 * Copyright (C) 1997  C. Scott Ananian <cananian@alumni.princeton.edu>
 * Copyright (c) 1998-1999 Bruce Kalk <kall@compass.com>
 *
 * Currently maintained by: Bruce Kall, <kall@compass.com>
 * 
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *    
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *    
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   $Id: tpconfig.c,v 2.13 2002/02/22 20:33:11 bruce Exp $
 */

#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <assert.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include <getopt.h>
#include <termios.h>

#ifdef __linux
#include <linux/ioctl.h>
#include <errno.h>
#define AUX_GRAB _IO('M', 1)
#endif

#include "tpconfig.h"

extern void synaptics_usage(char *progname);
extern void set_firmware_options(int fd, FILE *out);
extern void ALPS_usage(char *progname);
extern void synaptics_functions(int c,int fd,char **argv);
extern void alps_functions(int c,int fd,char **argv);
extern int is_ALPS(int fd);
extern int is_Synaptics(int fd);

int DEBUG_LEVEL;
int silent;
float firmware_rev;
char single_mode_byte;
int touchpad_type;

/* initialize file description if hasn't already been done. */
void init_fd(int *fd)
{ /* init_fd*/
int status;
if (*fd<0)
  {
  *fd = open("/dev/psaux", O_RDWR|O_NDELAY);
  if (*fd < 0)
    fatal("Could not open PS/2 Port [/dev/psaux]");
#ifdef __linux
  /* If AUX_GRAB kernel patch present, use it. */
  while (ioctl (*fd, AUX_GRAB, 0) < 0)
    {
      if (errno == ENOTTY)
	break;
      if (errno != EAGAIN)
	{
	  perror ("/dev/psaux");
	  fatal ("Unable to get exclusive access to PS/2 Port.");
	  exit (1);
	}
    }
#endif
  tcflush(*fd,TCIOFLUSH);
  if (!is_Synaptics(*fd))
    {
    status = is_ALPS(*fd);
    if(status == 0)
      {
      fatal("\n\nNo Synaptics or ALPS touchpad device found\n\n");
      exit(-1);
      }
    else
      {
      if(touchpad_type == ALPS_GLIDEPAD)
        printf ("Found ALPS GlidePad.\n");
      else if (touchpad_type == ALPS_STICKPOINTER_AND_GLIDEPOINT)
        printf ("Found ALPS Dual StickPoint/Glidepad.\n");
      else
        {
        fprintf (stderr, "Undefined touchpad found[%d]\n", touchpad_type);
        exit(-1);
        }
      }
    }
  else
    {
    printf ("Found Synaptics Touchpad.\n");
    touchpad_type = SYNAPTICS_TOUCHPAD;
    }
  }
} /* init_fd*/
 
 


int main(int argc, char **argv) 
{ /* main */
int fd = -1;
int c = 0;
int option_index = 0;

silent = 0;
firmware_rev = 0.0;
single_mode_byte = TRUE;

DEBUG_LEVEL = DEBUG_NONE;
#ifdef DEBUG
DEBUG_LEVEL = DEBUG_LOW;
#endif

copyright();


/* Open file descriptor and determine if we are connected to a touchpad */
init_fd(&fd);

if(touchpad_type == SYNAPTICS_TOUCHPAD)
  set_firmware_options(fd,stdout);

if((argc <  2)  && (touchpad_type == SYNAPTICS_TOUCHPAD))
  synaptics_usage(argv[0]); /* no command line options */
else if((argc <  2)  && 
    ((touchpad_type == ALPS_GLIDEPAD) || (touchpad_type == ALPS_STICKPOINTER_AND_GLIDEPOINT)))
  ALPS_usage(argv[0]);

opterr=1;
while (1) 
  {
  static struct option synaptics_long_options[] =
    {
      {"help",          no_argument, NULL, 'h'},
      {"silent",        no_argument, NULL, 'q'}, /* -q */
      {"quiet",         no_argument, NULL, 'q'}, /* -q */
      {"info",          no_argument, NULL, 'i'}, /* -i */
      {"version",       no_argument, NULL, 'v'}, /* -v */
      {"reset",         no_argument, NULL, 'x'}, /* -x */
      /* Mode 1 options */
      {"zthreshold",    optional_argument, NULL, 'z'}, /* -z */
      {"threshold",     optional_argument, NULL, 'z'}, /* -z */
      {"corner",        optional_argument, NULL, 'c'}, /* -c */
      {"tapmode",       optional_argument, NULL, 't'}, /* -t */
      {"edgemode",      optional_argument, NULL, 'e'}, /* -e */
      /* Mode 2 options */
      {"absolute",      no_argument, NULL, 'A'},       /* similar to -a */
      {"relative",      no_argument, NULL, 'R'},       /* similar to -a */
      {"rate",          optional_argument, NULL, 'r'}, /* -r */
      {"two-button",    no_argument, NULL, '2'},       /* -2 */
      {"three-button",  no_argument, NULL, '3'},       /* -3 */
      {"middle-button", no_argument, NULL, '<'},       /* similar to -m */
      {"right-button",  no_argument, NULL, '>'},       /* similar to -m */
      {"sleep",         optional_argument, NULL, 's'}, /* -s */
      /* Mode 3 options */
      {"right-margin",  optional_argument, NULL, '4'}, /* similar to -M */
      {"left-margin",   optional_argument, NULL, '5'}, /* similar to -M */
      /* Mode 4 options */
      {"top-margin",    optional_argument, NULL, '6'}, /* similar to -M */
      {"bottom-margin", optional_argument, NULL, '7'}, /* similar to -M */
      {NULL,            no_argument, NULL, 0}
    };

  static struct option ALPS_long_options[] = 
    {
      {"help",		no_argument, NULL, 'h'},
      {"info",		no_argument, NULL, 'i'}, /* -i */
      {"reset",         no_argument, NULL, 'x'}, /* -x */
      {"tapmode",	optional_argument, NULL, 't'}, /* -t */
      {"version", 	no_argument, NULL, 'v'}, /* -v */
      {NULL, 		no_argument, NULL, 0}
    };

  if(touchpad_type == SYNAPTICS_TOUCHPAD)
    c = getopt_long (argc, argv, "qivzx::c::t::e::a::r::23m::s::M",
      synaptics_long_options, &option_index);
  else if((touchpad_type == ALPS_STICKPOINTER_AND_GLIDEPOINT) || (touchpad_type == ALPS_GLIDEPAD))
    c = getopt_long (argc, argv, "hirtv",
      ALPS_long_options, &option_index);

  if (c == -1)
    break;
  
  if(touchpad_type == SYNAPTICS_TOUCHPAD)
    synaptics_functions(c,fd,argv);
  else if((touchpad_type == ALPS_STICKPOINTER_AND_GLIDEPOINT) || (touchpad_type == ALPS_GLIDEPAD))
    alps_functions(c,fd,argv);
  }
if (optind < argc) 
  {
  fprintf(stderr, "Extra options: ");
  while (optind < argc)
    fprintf(stderr, "%s ", argv[optind++]);
  fprintf(stderr, "\n");
  }
if (!(fd<0)) 
  close(fd);
return 1;
} /* main */



