@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15
#
# ecaccess-file-put: Upload a File on the ECaccess File System
#
# Laurent.Gougeon@ecmwf.int - 2010-10-15

use ECMWF::ECaccess;
use Getopt::Long;
use Pod::Usage;
use MIME::Base64;
use File::Basename;

my %opt = ( progress => 0, umask => 640, offset => 0, bufsize => 10485760, version => 0, help => 0, manual => 0, retry => 0, debug => 0 );

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if !GetOptions(
	\%opt,
	qw(
	  progress
	  umask=i
	  offset=i
	  bufsize=i
	  version
	  help|?
	  manual
	  retry=i
	  debug
	  )
);

# Display version if requested
die ECMWF::ECaccess->VERSION . "\n" if ( $opt{version} );

my $sourceLocalFile    = $ARGV[0];
my $targetECaccessFile = $ARGV[1];

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if ( $opt{help} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 2 ) if ( $opt{manual} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No source-local-file specified!\n" )        if not($sourceLocalFile);
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Source-local-file is not a plain file!\n" ) if not( -f $sourceLocalFile );

# Create the ECaccess Controler
my $ecaccess = ECMWF::ECaccess->new( $opt{retry}, $opt{debug});

# If no target is specified then take the source filename
$targetECaccessFile = basename($sourceLocalFile) if not($targetECaccessFile);

# Get the Token (using the Certificate in $HOME)
my $token = $ecaccess->getToken();

# Get the Control Channel
my $controlChannel = $ecaccess->getControlChannel();

# Get the file handle
$handle = $controlChannel->getOutputFileHandle( $token, $targetECaccessFile, $opt{offset}, $opt{umask} )->result;

# Open the Source File
open FILE, $sourceLocalFile or die "Error opening file: " . $sourceLocalFile . "\n";

# Progress bar
my $progressbar;
my $readCount = 0;
my $readTotal = 0;
if ( $opt{progress} && not( $^O =~ /^MSWin/ ) ) {
	eval "use Term::ProgressBar";
	$progressbar = Term::ProgressBar->new( { count => -s $sourceLocalFile, remove => 1 } );
	$progressbar->update(0);
}

# Upload the content
while ( ( $readCount = read( FILE, $data, $opt{bufsize} ) ) > 0 ) {
	$ecaccess->writeFileOutputStream( $handle, $data );
	$progressbar->update( $readTotal += $readCount ) if ( $opt{progress} && not( $^O =~ /^MSWin/ ) );
}

# close the file handles
$controlChannel->closeHandle($handle);
close FILE;

# Logout
$ecaccess->releaseToken($token);

__END__

=head1 NAME

ecaccess-file-put - Upload a File on the ECaccess File System

=head1 SYNOPSIS

B<ecaccess-file-put -version|-help|-manual>

B<ecaccess-file-put [-debug] [-progress] [-umask> I<mode>B<] [-offset> I<number>B<] [-bufsize> I<length>B<]> I<source-local-file> B<[>I<target-ecaccess-file>B<]>

=head1 DESCRIPTION

Allow uploading I<source-local-file> on the ECaccess File System. If no I<target-ecaccess-file> is specified then the
I<source-local-file> name is used.

The I<target-ecaccess-file> is in the form [domain:][/user-id/]path. Please read the "Shell commands -> File Management"
section of the "ecaccess" guide for more information on the ECaccess File System.

=head1 ARGUMENTS

=over 8

=item I<source-local-file>

The name of the source Local File.

=item I<target-ecaccess-file> (optional)

The name of the target ECaccess File.

=back

=head1 OPTIONS

=over 8

=item B<-progress>

Provide a progress meter on the standard terminal, allowing to monitor the file
transmission in real-time. The progress bar is removed from the terminal when
the transmission is done. This option is not supported and ignored on Windows
platforms.

=item B<-umask> I<mode>

The user file-creation I<mode> mask (umask) is used to determine the file permission
for newly created files. The default value is 640.

=item B<-offset> I<number>

Every open file has an associated file offset, which determines where the next
read operation will start. This I<number> is set to 0 by default.

=item B<-bufsize> I<length>

Specify the I<length> of the buffer (in bytes) which is used to upload the file.
The larger the buffer the smaller the number of http/s requests. By default a
buffer of 10485760 bytes (10MB) is used.

=item B<-version>

Display version number and exits.

=item B<-help>

Print a brief help message and exits.

=item B<-manual>

Prints the manual page and exits.

=item B<-retry> I<count>

Number of SSL connection retries per 5s to ECMWF. This parameter only apply to the
initial SSL connection initiated by the command to the ECMWF server. It does not
apply to all the subsequent requests made afteward as it is mainly targeting errors
that can happen from time to time during the SSL handshake. Default is no retry.

=item B<-debug>

Display the SOAP and SSL messages exchanged.

=back

=head1 EXAMPLES

B<ecaccess-file-put> I<$HOME/bin/a.out>

Upload the local I<$HOME/bin/a.out> File in the $HOME directory of the authenticated user. The Target File Name is I<a.out>.

B<ecaccess-file-put> I<$HOME/bin/a.out> I<c1a:/tmp/a2.out>

Upload the local I<$HOME/bin/a.out> File in the tmp directory of c1a. The Target File Name is I<a2.out>.

=head1 SEE ALSO

B<ecaccess-file-delete>, B<ecaccess-file-get>, B<ecaccess-file-mget>, B<ecaccess-file-modtime>, B<ecaccess-file-mput>,
B<ecaccess-file-rmdir>, B<ecaccess-file-copy>, B<ecaccess-file-dir>, B<ecaccess-file-mdelete>, B<ecaccess-file-mkdir>,
B<ecaccess-file-move>, B<ecaccess-file-chmod>, B<ecaccess-file-size> and B<ecaccess>.

=cut

__END__
:endofperl
