------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-strings.adb,v 1.27 2008/07/03 06:22:48 vagul Exp $

with Ada.Exceptions;
with Ada.Text_IO;
with Ada.Unchecked_Deallocation;

with OCI.Lib;
with OCI.Thread;

with System;

package body OCI.Thick.Strings is
   use type Sb2;

   Debug_Mode : constant Boolean := False;

   First_Block_Length : constant := 64;
   Next_Block_Length  : constant := 1024;

   type Access_Size_T is access all size_t;

   subtype Buffer_Access is System.Address;

   Null_Buffer : constant Buffer_Access := System.Null_Address;

   type A_A_Ub2 is access A_Ub2;
   type A_A_Sb2 is access A_Sb2;

   --  Type_Id : constant := OCI.Lib.SQLT_LNG; -- For 8.1.7
   --  ??? Do not use OCI.Lib.SQLT_LNG type code becouse dynamic define
   --  under Oracle 9.x.x do not work correctly with it.

   --  Type_Id : constant := OCI.Lib.SQLT_CHR; -- For 9.xxx.
   --  ??? Do not use OCI.Lib.SQLT_CHR type code under Oracle 8.1.x becouse
   --  dynamic define do not work correctly with it.

   Type_Id : constant := OCI.Lib.SQLT_STR;
   --  Looks like both 8i and 9i oracle working reasonably with it.

   Nul_Terminated : constant Boolean := Type_Id = OCI.Lib.SQLT_STR;

   String_Max_Size : constant := 32767;

   procedure Debug_Print (Str : String);
   pragma Inline (Debug_Print);

   procedure Print_Error (Str : String);

   procedure Free is new Ada.Unchecked_Deallocation
                           (char_array, Char_Array_Access);

   procedure Bind_Dynamic (Value : in out Var_Type);
   pragma Inline (Bind_Dynamic);

   procedure Complete (Item : in Var_Type);

   procedure Debug_Print
     (Prefix : in String;
      alenp  : in Access_Size_T;
      piece  : in Ub1;
      indp   : in A_Sb2;
      rcodep : in A_Ub2);
   pragma Inline (Debug_Print);

   function Callback_In_Bind
     (ictxp  : in     Value_Access;
      bindp  : in     OCIBind;
      iter   : in     Ub4;
      index  : in     Ub4;
      bufpp  : access Buffer_Access;
      alenp  : in     Access_Size_T;
      piecep : in     A_Ub1;
      indpp  : in     A_A_Sb2)
      return SWord;
   pragma Convention (C, Callback_In_Bind);

   function Callback_Out_Bind
     (octxp   : in     Value_Access;
      bindp   : in     OCIBind;
      iter    : in     Ub4;
      index   : in     Ub4;
      bufpp   : access Buffer_Access;
      alenpp  : access Access_Size_T;
      piecep  : in     A_Ub1;
      indpp   : in     A_A_Sb2;
      rcodepp : in     A_A_Ub2)
      return  SWord;
   pragma Convention (C, Callback_Out_Bind);

   procedure Callback_Out
     (Value  : in     Value_Access;
      bufp   : in out Buffer_Access;
      alenp  : in out Access_Size_T;
      piece  : in     Ub1;
      indp   : in out A_Sb2;
      rcodep : in out A_Ub2;
      Result :    out SWord);

   function Callback_Define
     (octxp   : in     Value_Access;
      defnp   : in     OCIDefine;
      iter    : in     Ub4;
      bufpp   : access Buffer_Access;
      alenpp  : access Access_Size_T;
      piecep  : in     A_Ub1;
      indpp   : in     A_A_Sb2;
      rcodepp : in     A_A_Ub2)
      return  SWord;
   pragma Convention (C, Callback_Define);

   use type Sb4;

   ----------
   -- Bind --
   ----------

   procedure Bind
     (Stmt  : in     Statement'Class;
      Value : in out Var_Type;
      Name  : in     String;
      Exist :    out Boolean)
   is
      RC : constant SWord := OCIBindByName
                  (Stmtp       => OCIStmt (Handle (Stmt)),
                   Bindpp      => Value.Bind'Access,
                   Errhp       => OCI.Thread.Error,
                   Placeholder => C.To_C (Name),
                   Placeh_Len  => Name'Length,
                   Valuep      => System.Null_Address,
                   Value_Sz    => String_Max_Size,
                   Dty         => Type_Id,
                   Indp        => null,
                   Mode        => OCI_DATA_AT_EXEC);
   begin
      Exist := not (RC = OCI_ERROR and then Last_Error_Code = 1036);

      if Exist then
         Check_Error (RC);
         Bind_Dynamic (Value);
      end if;

      if Debug_Mode then
         Value.Name := To_Unbounded_String (Name);
      end if;
   end Bind;

   procedure Bind
     (Stmt  : in     Statement'Class;
      Value : in out Var_Type;
      Name  : in     String) is
   begin
      Check_Error
        (OCIBindByName
           (Stmtp       => OCIStmt (Handle (Stmt)),
            Bindpp      => Value.Bind'Access,
            Errhp       => OCI.Thread.Error,
            Placeholder => C.To_C (Name),
            Placeh_Len  => Name'Length,
            Valuep      => System.Null_Address,
            Value_Sz    => String_Max_Size,
            Dty         => Type_Id,
            Indp        => null,
            Mode        => OCI_DATA_AT_EXEC));

      Bind_Dynamic (Value);

      if Debug_Mode then
         Value.Name := To_Unbounded_String (Name);
      end if;
   end Bind;

   procedure Bind
     (Stmt     : in     Statement'Class;
      Value    : in out Var_Type;
      Position : in     Positive) is
   begin
      Check_Error
        (OCIBindByPos
           (Stmtp    => OCIStmt (Handle (Stmt)),
            Bindpp   => Value.Bind'Access,
            Errhp    => OCI.Thread.Error,
            Position => Ub4 (Position),
            Valuep   => System.Null_Address,
            Value_Sz => String_Max_Size,
            Dty      => Type_Id,
            Indp     => null,
            Mode     => OCI_DATA_AT_EXEC));
      Bind_Dynamic (Value);
   end Bind;

   ------------------
   -- Bind_Dynamic --
   ------------------

   procedure Bind_Dynamic (Value : in out Var_Type) is
   begin
      Check_Error (OCIBindDynamic
        (Bindp  => Value.Bind,
         Errhp  => OCI.Thread.Error,
         Ictxp  => Value'Address,
         Icbfp  => Callback_In_Bind'Address,
         Octxp  => Value'Address,
         Ocbfp  => Callback_Out_Bind'Address));
   end Bind_Dynamic;

   ---------------------
   -- Callback_Define --
   ---------------------

   function Callback_Define
     (octxp   : in     Value_Access;
      defnp   : in     OCIDefine;
      iter    : in     Ub4;
      bufpp   : access Buffer_Access;
      alenpp  : access Access_Size_T;
      piecep  : in     A_Ub1;
      indpp   : in     A_A_Sb2;
      rcodepp : in     A_A_Ub2)
      return SWord
   is
      pragma Unreferenced (defnp, iter);
      Result : SWord;
   begin
      Callback_Out
        (Value  => octxp,
         bufp   => bufpp.all,
         alenp  => alenpp.all,
         piece  => piecep.all,
         indp   => indpp.all,
         rcodep => rcodepp.all,
         Result => Result);

      return Result;
   end Callback_Define;

   ----------------------
   -- Callback_In_Bind --
   ----------------------

   function Callback_In_Bind
     (ictxp  : in     Value_Access;
      bindp  : in     OCIBind;
      iter   : in     Ub4;
      index  : in     Ub4;
      bufpp  : access Buffer_Access;
      alenp  : in     Access_Size_T;
      piecep : in     A_Ub1;
      indpp  : in     A_A_Sb2)
      return SWord
   is
      Item  : Value_Access renames ictxp;

      pragma Unreferenced (bindp, iter, index);

   begin
      Complete (Item.all);

      Debug_Print
        (Prefix => "in_ " & To_String (Item.Name),
         alenp  => alenp,
         piece  => piecep.all,
         indp   => indpp.all,
         rcodep => null);

      indpp.all  := Item.Indicator'Access;
      piecep.all := OCI_ONE_PIECE;

      if Item.Indicator = Null_Indicator then
         alenp.all := 0;
         bufpp.all := Null_Buffer;
      else
         if Nul_Terminated
           and then Item.Value /= null
           and then Item.Value (Item.alen - 1) /= nul
         then
            if Item.Value'Length > Item.alen then
               Item.Value (Item.alen) := nul;
            elsif Item.Value'Length = Item.alen then
               declare
                  Ptr : constant Char_Array_Access
                    := new char_array'(Item.Value.all & nul);
               begin
                  Free (Item.Value);
                  Item.Value := Ptr;
               end;
            else
               Print_Error ("In bind variable length error.");

               return OCI_ERROR;
            end if;

            Item.alen := Item.alen + 1;
         end if;

         alenp.all := Item.alen;
         bufpp.all := Item.Value.all'Address;
      end if;

      Debug_Print
        (Prefix => "_in " & To_String (Item.Name),
         alenp  => alenp,
         piece  => piecep.all,
         indp   => indpp.all,
         rcodep => null);

      return OCI_CONTINUE;
   exception
      when E : others =>
         Print_Error (Ada.Exceptions.Exception_Information (E));
         return OCI_ERROR;
   end Callback_In_Bind;

   ------------------
   -- Callback_Out --
   ------------------

   procedure Callback_Out
     (Value  : in     Value_Access;
      bufp   : in out Buffer_Access;
      alenp  : in out Access_Size_T;
      piece  : in     Ub1;
      indp   : in out A_Sb2;
      rcodep : in out A_Ub2;
      Result :    out SWord) is
   begin
      Debug_Print
        (Prefix => "out_ " & To_String (Value.Name),
         alenp  => alenp,
         piece  => piece,
         indp   => indp,
         rcodep => rcodep);

      alenp  := Value.alen'Access;
      indp   := Value.Indicator'Access;
      rcodep := Value.rcode'Access;

      case piece is
      when OCI_ONE_PIECE | OCI_FIRST_PIECE =>
         Value.Self := Value;

         if Value.Value = null then
            Value.Value := new char_array (0 .. First_Block_Length - 1);

            --  ??? For new allocated array put first byte to zero.
            --  Try to detect unwrited buffer.

            Value.Value (0) := nul;
         end if;

         --  To be sure that it was a first time.

         Free (Value.Next);

         bufp       := Value.Value.all'Address;
         Value.alen := Value.Value'Length;

      when OCI_NEXT_PIECE =>
         if Value.Next = null then
            Value.Next := new char_array (0 .. Next_Block_Length - 1);
         else
            declare
               Ptr : constant Char_Array_Access
                  := new char_array'(Value.Value.all & Value.Next.all);
            begin
               Free (Value.Value);
               Value.Value := Ptr;
            end;
         end if;

         Value.alen := Value.Next'Length;
         bufp       := Value.Next.all'Address;

      when OCI_LAST_PIECE =>
         Complete (Value.all);
      when others =>
         Print_Error ("piece = " & Ub1'Image (piece));
         Result := OCI_ERROR;
      end case;

      Debug_Print
        (Prefix => "_out " & To_String (Value.Name),
         alenp  => alenp,
         piece  => piece,
         indp   => indp,
         rcodep => rcodep);

      Result := OCI_CONTINUE;
   exception
      when E : others =>
         Print_Error
           ("In callback " & Ada.Exceptions.Exception_Information (E));

         Result := OCI_ERROR;
   end Callback_Out;

   -----------------------
   -- Callback_Out_Bind --
   -----------------------

   function Callback_Out_Bind
     (octxp   : in     Value_Access;
      bindp   : in     OCIBind;
      iter    : in     Ub4;
      index   : in     Ub4;
      bufpp   : access Buffer_Access;
      alenpp  : access Access_Size_T;
      piecep  : in     A_Ub1;
      indpp   : in     A_A_Sb2;
      rcodepp : in     A_A_Ub2)
      return  SWord
   is
      pragma Unreferenced (bindp, iter, index);

      Result : SWord;
   begin
      Callback_Out
        (Value  => octxp,
         bufp   => bufpp.all,
         alenp  => alenpp.all,
         piece  => piecep.all,
         indp   => indpp.all,
         rcodep => rcodepp.all,
         Result => Result);

      return Result;
   end Callback_Out_Bind;

   -----------------
   -- Clear_Value --
   -----------------

   procedure Clear_Value (Var : in out Var_Type) is
   begin
      Var.alen := 0;
      Free (Var.Value);
      Free (Var.Next);

      Clear_Value (Inherited (Var));
   end Clear_Value;

   --------------
   -- Complete --
   --------------

   procedure Complete (Item : in Var_Type) is
   begin
      if Item.Self = null then
         return;
      end if;

      Debug_Print
        (Prefix => "end_ " & To_String (Item.Name),
         alenp  => Item.Self.alen'Access,
         piece  => 100,
         indp   => Item.Self.Indicator'Access,
         rcodep => Item.Self.rcode'Access);

      if Item.Next /= null then
         declare
            Ptr  : Char_Array_Access;
         begin
            if Nul_Terminated and Item.Next (Item.alen - 1) /= nul then
               Ptr := new char_array'(Item.Value.all
                                      & Item.Next (0 .. Item.alen - 1)
                                      & nul);
               Debug_Print ("Result was not null terminated.");
            else
               Ptr := new char_array'(Item.Value.all
                                      & Item.Next (0 .. Item.alen - 1));
            end if;

            Free (Item.Self.Value);
            Free (Item.Self.Next);

            Item.Self.Value := Ptr;
            Item.Self.alen  := Ptr'Length;
         end;

      elsif Item.alen = 0 then
         Free (Item.Self.Value);

         if Item.Indicator /= Null_Indicator then
            Debug_Print ("wrong null indicator" & Item.Indicator'Img);
            Item.Self.Indicator := Null_Indicator;
         end if;

      elsif Item.Value (0) = nul and then Item.Indicator = Null_Indicator then
         Free (Item.Self.Value);
         Item.Self.alen := 0;
      end if;

      if Item.alen /= 0 and then Item.Indicator = Null_Indicator then
         Debug_Print ("wrong not null indicator" & Item.Indicator'Img);
         Item.Self.Indicator := Not_Null_Indicator;
      end if;

      Debug_Print
        (Prefix => "_end " & To_String (Item.Name),
         alenp  => Item.Self.alen'Access,
         piece  => 100,
         indp   => Item.Self.Indicator'Access,
         rcodep => Item.Self.rcode'Access);

      Item.Self.Self := null;
   end Complete;

   -----------------
   -- Debug_Print --
   -----------------

   procedure Debug_Print (Str : String) is
   begin
      if Debug_Mode then
         Ada.Text_IO.Put_Line ("# " & Str);
      end if;
   end Debug_Print;

   procedure Debug_Print
     (Prefix : in String;
      alenp  : in Access_Size_T;
      piece  : in Ub1;
      indp   : in A_Sb2;
      rcodep : in A_Ub2)
   is
      use Ada.Text_IO;
      use ASCII;
   begin
      if not Debug_Mode then
         return;
      end if;

      Put ("# " & Prefix & HT);

      if alenp = null then
         Put ("(null)");
      else
         Put (alenp.all'Img);
      end if;

      Put (HT);

      case piece is
      when OCI_ONE_PIECE   => Put ("ONE");
      when OCI_FIRST_PIECE => Put ("FIRST");
      when OCI_NEXT_PIECE  => Put ("NEXT");
      when OCI_LAST_PIECE  => Put ("LAST");
      when others          => Put (piece'Img);
      end case;

      Put (HT);

      if indp = null then
         Put ("(null)");
      else
         case indp.all is
         when Null_Indicator      => Put ("Null");
         when Not_Null_Indicator  => Put ("Full");
         when others              => Put (indp.all'Img);
         end case;
      end if;

      Put (HT);

      if rcodep = null then
         Put ("(null)");
      else
         Put (rcodep.all'Img);
      end if;

      New_Line;
   end Debug_Print;

   ------------
   -- Define --
   ------------

   procedure Define
     (Stmt     : in     Statement'Class;
      Value    : in out Var_Type;
      Position : in     Positive)
   is
      Rc : SWord := OCIDefineByPos
         (Stmtp    => OCIStmt (Handle (Stmt)),
          Defnpp   => Value.Define'Access,
          Errhp    => OCI.Thread.Error,
          Position => Ub4 (Position),
          Value    => System.Null_Address,
          Value_Sz => String_Max_Size,
          Dty      => Type_Id,
          Indp     => null,
          Mode     => OCI_DYNAMIC_FETCH);
   begin
      Check_Error (Rc);

      Rc := OCIDefineDynamic
        (Defnp  => Value.Define,
         Errhp  => OCI.Thread.Error,
         Octxp  => Value'Address,
         Ocbfp  => Callback_Define'Address);

      Check_Error (Rc);
   end Define;

   --------------
   -- Finalize --
   --------------

   procedure Finalize  (Object : in out Var_Type) is
      use type System.Address;
   begin
      Free (Object.Value);
      Free (Object.Next);

      OCI.Thick.Finalize (Limited_Variable (Object));
   end Finalize;

   ---------------
   -- Get_Value --
   ---------------

   function Get_Value (Var : Var_Type) return Unbounded_String is
   begin
      return To_Unbounded_String (Get_Value (Var));
   end Get_Value;

   function Get_Value
     (Var     : in Var_Type;
      Default : in String;
      Mode    : in Default_Mode := Instead_Any)
      return  String is
   begin
      Complete (Var);

      if not Is_Attached (Var) then
         if Mode = Instead_Not_Attached or Mode = Instead_Any then
            return Default;
         else
            raise Not_Attached;
         end if;
      elsif Is_Null (Var) then
         if Mode = Instead_Null or Mode = Instead_Any then
            return Default;
         else
            raise Null_Value;
         end if;
      else
         return To_Ada (Var.Value (0 .. Var.alen - 1)  & nul);
      end if;
   end Get_Value;

   function Get_Value (Var : Var_Type) return String is
   begin
      Complete (Var);

      if Is_Null (Var) then
         raise Null_Value;
      end if;

      return To_Ada (Var.Value (0 .. Var.alen - 1)  & nul);
   end Get_Value;

   -------------
   -- Is_Null --
   -------------

   function Is_Null (Var : Var_Type) return Boolean is
   begin
      Complete (Var);

      return Is_Null (Inherited (Var));
   end Is_Null;

   -----------------
   -- Print_Error --
   -----------------

   procedure Print_Error (Str : String) is
      use Ada.Text_IO;
   begin
      Put_Line (Current_Error, Str);
   end Print_Error;

   ---------------
   -- Set_Value --
   ---------------

   procedure Set_Value (Var : in out Var_Type; Value : Unbounded_String) is
   begin
      Set_Value (Var, To_String (Value));
   end Set_Value;

   procedure Set_Value (Var : in out Var_Type; Value : String) is
   begin
      Free (Var.Value);
      Free (Var.Next);

      if Value = "" then
         Clear_Value (Var);
      else
         Var.Indicator := Not_Null_Indicator;
         Var.Value := new char_array'
                            (To_C
                                (Value,
                                 Append_Nul => Nul_Terminated));

         Var.alen  := Var.Value'Length;
      end if;
   end Set_Value;

end OCI.Thick.Strings;
