/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     lagrange_2_3d.c                                                */
/*                                                                          */
/* description:  piecewise quadratic Lagrange elements in 2d                */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#define N_BAS2_3D  10

static const REAL   bary2_3d[N_BAS2_3D][N_LAMBDA] = {{1.0, 0.0, 0.0, 0.0},
						     {0.0, 1.0, 0.0, 0.0},
						     {0.0, 0.0, 1.0, 0.0},
						     {0.0, 0.0, 0.0, 1.0},
						     {0.5, 0.5, 0.0, 0.0},
						     {0.5, 0.0, 0.5, 0.0},
						     {0.5, 0.0, 0.0, 0.5},
						     {0.0, 0.5, 0.5, 0.0},
						     {0.0, 0.5, 0.0, 0.5},
						     {0.0, 0.0, 0.5, 0.5}};

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 0                                               */
/*--------------------------------------------------------------------------*/

static REAL phi2v0_3d(const REAL lambda[N_LAMBDA])
{
  return(lambda[0]*(2.0*lambda[0] - 1.0));
}

static const REAL *grd_phi2v0_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};
  grd[0] = 4.0*lambda[0] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi2v0_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{4, 0, 0, 0}, {0, 0, 0, 0}, 
					{0, 0, 0, 0}, {0, 0, 0, 0}};
  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 1                                               */
/*--------------------------------------------------------------------------*/

static REAL phi2v1_3d(const REAL lambda[N_LAMBDA])
{
  return(lambda[1]*(2.0*lambda[1] - 1.0));
}

static const REAL *grd_phi2v1_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};
  grd[1] = 4.0*lambda[1] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi2v1_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0}, {0, 4, 0, 0}, 
					{0, 0, 0, 0}, {0, 0, 0, 0}};
  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 2                                               */
/*--------------------------------------------------------------------------*/

static REAL phi2v2_3d(const REAL lambda[N_LAMBDA])
{
  return(lambda[2]*(2.0*lambda[2] - 1.0));
}

static const REAL *grd_phi2v2_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};
  grd[2] = 4.0*lambda[2] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi2v2_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0}, {0, 0, 0, 0}, 
					{0, 0, 4, 0}, {0, 0, 0, 0}};
  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 3                                               */
/*--------------------------------------------------------------------------*/

static REAL phi2v3_3d(const REAL lambda[N_LAMBDA])
{
  return(lambda[3]*(2.0*lambda[3] - 1.0));
}

static const REAL *grd_phi2v3_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};
  grd[3] = 4.0*lambda[3] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*D2_phi2v3_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0}, {0, 0, 0, 0}, 
					{0, 0, 0, 0}, {0, 0, 0, 4}};
  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at edge 0                                                 */
/*--------------------------------------------------------------------------*/

static REAL phi2e0_3d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[0]*lambda[1]);
}

static const REAL *grd_phi2e0_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};
  grd[0] = 4.0*lambda[1];
  grd[1] = 4.0*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi2e0_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 4, 0, 0}, {4, 0, 0, 0}, 
					{0, 0, 0, 0}, {0, 0, 0, 0}};
  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at edge 1                                                 */
/*--------------------------------------------------------------------------*/

static REAL phi2e1_3d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[0]*lambda[2]);
}

static const REAL *grd_phi2e1_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};
  grd[0] = 4.0*lambda[2];
  grd[2] = 4.0*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi2e1_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 4, 0}, {0, 0, 0, 0}, 
					{4, 0, 0, 0}, {0, 0, 0, 0}};
  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at edge 2                                                 */
/*--------------------------------------------------------------------------*/

static REAL phi2e2_3d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[0]*lambda[3]);
}

static const REAL *grd_phi2e2_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};
  grd[0] = 4.0*lambda[3];
  grd[3] = 4.0*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi2e2_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 4}, {0, 0, 0, 0}, 
					{0, 0, 0, 0}, {4, 0, 0, 0}};
  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at edge 3                                                 */
/*--------------------------------------------------------------------------*/

static REAL phi2e3_3d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[1]*lambda[2]);
}

static const REAL *grd_phi2e3_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};
  grd[1] = 4.0*lambda[2];
  grd[2] = 4.0*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi2e3_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0}, {0, 0, 4, 0}, 
					{0, 4, 0, 0}, {0, 0, 0, 0}};
  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at edge 4                                                 */
/*--------------------------------------------------------------------------*/

static REAL phi2e4_3d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[1]*lambda[3]);
}

static const REAL *grd_phi2e4_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};
  grd[1] = 4.0*lambda[3];
  grd[3] = 4.0*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi2e4_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0}, {0, 0, 0, 4}, 
					{0, 0, 0, 0}, {0, 4, 0, 0}};
  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at edge 5                                                 */
/*--------------------------------------------------------------------------*/

static REAL phi2e5_3d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[2]*lambda[3]);
}

static const REAL *grd_phi2e5_3d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};
  grd[2] = 4.0*lambda[3];
  grd[3] = 4.0*lambda[2];
  return((const REAL *) grd);
}

static const REAL (*D2_phi2e5_3d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0}, {0, 0, 0, 0}, 
					      {0, 0, 0, 4}, {0, 0, 4, 0}};
  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  functions for combining basisfunctions with coefficients                */
/*--------------------------------------------------------------------------*/

static const DOF *get_dof_indices2_3d(const EL *el, const DOF_ADMIN *admin,
				      DOF *idof)
{
  static DOF  index_vec[N_VERTICES_3D+N_EDGES_3D];
  int         i, n0;
  DOF         *rvec = idof ? idof : index_vec;
  DOF         **dof = el->dof;
  n0 = admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_3D; i++)
    rvec[i] = dof[i][n0];
  n0 = admin->n0_dof[EDGE];
  for (i = N_VERTICES_3D; i < N_VERTICES_3D+N_EDGES_3D; i++)
    rvec[i] = dof[i][n0];

  return((const DOF *) rvec);
}

static const S_CHAR *get_bound2_3d(const EL_INFO *el_info, S_CHAR *bound)
{
  FUNCNAME("get_bound2_3d");
  static S_CHAR  bound_vec[N_VERTICES_3D+N_EDGES_3D];
  S_CHAR         *rvec = bound ? bound : bound_vec;
  int            i;

  DEBUG_TEST_FLAG(FILL_BOUND, el_info);

  for (i = 0; i < N_VERTICES_3D; i++)
    rvec[i] = el_info->vertex_bound[i];
  for (i = 0; i < N_EDGES_3D; i++)
    rvec[N_VERTICES_3D+i] = el_info->edge_bound[i];

  return((const S_CHAR *) rvec);
}

static const int *get_int_vec2_3d(const EL *el, const DOF_INT_VEC *vec,
				  int *ivec)
{
  FUNCNAME("get_int_vec2_3d");
  int           i, n0;
  static int    local_vec[N_VERTICES_3D+N_EDGES_3D];
  int           *v = nil, *rvec = ivec ? ivec : local_vec;
  DOF           **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_3D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = N_VERTICES_3D; i < N_VERTICES_3D+N_EDGES_3D; i++)
    rvec[i] = v[dof[i][n0]];

  return((const int *) rvec);
}

static const REAL *get_real_vec2_3d(const EL *el, const DOF_REAL_VEC *vec,
				    REAL *Rvec)
{
  FUNCNAME("get_real_vec2_3d");
  int            i, n0;
  static REAL    local_vec[N_VERTICES_3D+N_EDGES_3D];
  REAL           *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_3D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = N_VERTICES_3D; i < N_VERTICES_3D+N_EDGES_3D; i++)
    rvec[i] = v[dof[i][n0]];

  return((const REAL *) rvec);
}

static const REAL_D *get_real_d_vec2_3d(const EL *el, 
					const DOF_REAL_D_VEC *vec,
					REAL_D *Rvec)
{
  FUNCNAME("get_real_d_vec2_3d");
  int            i, k, n0;
  static REAL_D  local_vec[N_BAS2_3D];
  REAL_D         *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_3D; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[i][k] = v[dof[i][n0]][k];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = N_VERTICES_3D; i < N_VERTICES_3D+N_EDGES_3D; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[i][k] = v[dof[i][n0]][k];

  return((const REAL_D *) rvec);
}

static const U_CHAR *get_uchar_vec2_3d(const EL *el, const DOF_UCHAR_VEC *vec,
				       U_CHAR *uvec)
{
  FUNCNAME("get_uchar_vec2_3d");
  int            i, n0;
  static U_CHAR  local_vec[N_VERTICES_3D+N_EDGES_3D];
  U_CHAR         *v = nil, *rvec = uvec ? uvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_3D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = N_VERTICES_3D; i < N_VERTICES_3D+N_EDGES_3D; i++)
    rvec[i] = v[dof[i][n0]];

  return((const U_CHAR *) rvec);
}

static const S_CHAR *get_schar_vec2_3d(const EL *el, const DOF_SCHAR_VEC *vec,
				       S_CHAR *svec)
{
  FUNCNAME("get_schar_vec2_3d");
  int            i, n0;
  static S_CHAR  local_vec[N_VERTICES_3D+N_EDGES_3D];
  S_CHAR         *v = nil, *rvec = svec ? svec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_3D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = N_VERTICES_3D; i < N_VERTICES_3D+N_EDGES_3D; i++)
    rvec[i] = v[dof[i][n0]];

  return((const S_CHAR *) rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of scalar functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL(,2,3,10)

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of vector functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL_D(,2,3,10)

/*--------------------------------------------------------------------------*/
/*  functions for interpolation/ restriction during refinement/coarsening   */
/*--------------------------------------------------------------------------*/

static void real_refine_inter2_3d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_refine_inter2_3d");
  EL        *el;
  REAL      *v = nil;
  const DOF *cdof;
  DOF       pdof[N_BAS2_3D], cdofi;
  int       i, lr_set;
  int       node0, n0;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  get_dof_indices(el, admin, pdof);

  node0 = drv->fe_space->mesh->node[EDGE];
  n0 = admin->n0_dof[EDGE];

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[0], admin, nil);

  v[cdof[3]] = (v[pdof[4]]);
  v[cdof[6]] = (0.375*v[pdof[0]] - 0.125*v[pdof[1]] 
		+ 0.75*v[pdof[4]]);
  v[cdof[8]] = (0.125*(-v[pdof[0]] - v[pdof[1]]) + 0.25*v[pdof[4]]
		+ 0.5*(v[pdof[5]] + v[pdof[7]]));
  v[cdof[9]] = (0.125*(-v[pdof[0]] - v[pdof[1]]) + 0.25*v[pdof[4]]
		+ 0.5*(v[pdof[6]] + v[pdof[8]]));

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/
  
  cdofi = el->child[1]->dof[node0+2][n0];
  v[cdofi] = (-0.125*v[pdof[0]] + 0.375*v[pdof[1]] 
	      + 0.75*v[pdof[4]]);

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/
  
  for (i = 1; i < n; i++)
  {
    el = list[i].el_info.el;
    get_dof_indices(el, admin, pdof);

    lr_set = 0;
    if (list[i].neigh[0]  &&  list[i].neigh[0]->no < i)
      lr_set = 1;

    if (list[i].neigh[1]  &&  list[i].neigh[1]->no < i)
      lr_set += 2;

    DEBUG_TEST_EXIT(lr_set, "no values set on both neighbours\n");

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

    switch (lr_set)
    {
    case 1:
      cdofi = el->child[0]->dof[node0+4][n0];      
      v[cdofi] = (0.125*(-v[pdof[0]] - v[pdof[1]]) + 0.25*v[pdof[4]]
		  + 0.5*(v[pdof[5]] + v[pdof[7]]));
      break;
    case 2:
      cdofi = el->child[0]->dof[node0+5][n0];      
      v[cdofi] = (0.125*(-v[pdof[0]] - v[pdof[1]]) + 0.25*v[pdof[4]]
		  + 0.5*(v[pdof[6]] + v[pdof[8]]));
    }
  }
  return;
}

static void real_coarse_inter2_3d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_coarse_inter2_3d");
  EL        *el;
  REAL      *v = nil;
  int       cdof, pdof;
  const DOF_ADMIN *admin;
  MESH      *mesh = nil;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  GET_STRUCT(admin,drv->fe_space);
  GET_STRUCT(mesh,drv->fe_space);

  cdof = el->child[0]->dof[mesh->node[VERTEX]+3][admin->n0_dof[VERTEX]]; 
  pdof = el->dof[mesh->node[EDGE]][admin->n0_dof[EDGE]]; 
  v[pdof] = v[cdof];

  return;
}

static void real_coarse_restr2_3d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_coarse_restr2_3d");
  EL        *el;
  REAL      *v = nil;
  const DOF *cdof;
  DOF       pdof[N_BAS2_3D], cdofi;
  int       i, lr_set;
  int       node0, n0;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  get_dof_indices(el, admin, pdof);

  node0 = drv->fe_space->mesh->node[EDGE];
  n0 = admin->n0_dof[EDGE];

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[0], admin, nil);

  v[pdof[0]] += (0.375*v[cdof[6]] + 0.125*(-v[cdof[8]] - v[cdof[9]]));
  v[pdof[1]] += (0.125*(-v[cdof[6]] - v[cdof[8]] - v[cdof[9]]));
  v[pdof[4]] = (v[cdof[3]] + 0.75*v[cdof[6]] 
		+ 0.25*(v[cdof[8]] + v[cdof[9]]));
  v[pdof[5]] += (0.5*v[cdof[8]]);
  v[pdof[6]] += (0.5*v[cdof[9]]);
  v[pdof[7]] += (0.5*v[cdof[8]]);
  v[pdof[8]] += (0.5*v[cdof[9]]);

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[1], admin, nil);
  cdofi = el->child[1]->dof[node0+2][n0];

  v[pdof[0]] += (-0.125*v[cdofi]);
  v[pdof[1]] += (0.375*v[cdofi]);
  v[pdof[4]] += (0.75*v[cdofi]);

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/
  
  for (i = 1; i < n; i++)
  {
    el = list[i].el_info.el;
    get_dof_indices(el, admin, pdof);

    lr_set = 0;
    if (list[i].neigh[0]  &&  list[i].neigh[0]->no < i)
      lr_set = 1;

    if (list[i].neigh[1]  &&  list[i].neigh[1]->no < i)
      lr_set += 2;

    DEBUG_TEST_EXIT(lr_set, "no values set on both neighbours\n");

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

    cdof = get_dof_indices(el->child[0], admin, nil);

    switch(lr_set)
    {
    case 1:
      cdofi = el->child[0]->dof[node0+4][n0];      
      v[pdof[0]] += (-0.125*v[cdofi]);
      v[pdof[1]] += (-0.125*v[cdofi]);
      v[pdof[4]] += (0.25*v[cdofi]);
      v[pdof[5]] += (0.5*v[cdofi]);
      v[pdof[7]] += (0.5*v[cdofi]);
      break;
    case 2:
      cdofi = el->child[0]->dof[node0+5][n0];      
      v[pdof[0]] += (-0.125*v[cdofi]);
      v[pdof[1]] += (-0.125*v[cdofi]);
      v[pdof[4]] += (0.25*v[cdofi]);
      v[pdof[6]] += (0.5*v[cdofi]);
      v[pdof[8]] += (0.5*v[cdofi]);
      break;
    }
  }
  return;
}

static void real_d_refine_inter2_3d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list,
				    int n)
{
  FUNCNAME("real_d_refine_inter2_3d");
  EL        *el;
  REAL_D    *v = nil;
  const DOF *cd;
  DOF       pd[N_BAS2_3D], cdi;
  int       i, k, lr_set;
  int       node0, n0;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  get_dof_indices = drdv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drdv->fe_space);

  get_dof_indices(el, admin, pd);

  node0 = drdv->fe_space->mesh->node[EDGE];
  n0 = admin->n0_dof[EDGE];

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[0], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[cd[3]][k] = (v[pd[4]][k]);
    v[cd[6]][k] = (0.375*v[pd[0]][k] - 0.125*v[pd[1]][k] + 0.75*v[pd[4]][k]);
    v[cd[8]][k] = (0.125*(-v[pd[0]][k] - v[pd[1]][k]) + 0.25*v[pd[4]][k]
		   + 0.5*(v[pd[5]][k] + v[pd[7]][k]));
    v[cd[9]][k] = (0.125*(-v[pd[0]][k] - v[pd[1]][k]) + 0.25*v[pd[4]][k]
		   + 0.5*(v[pd[6]][k] + v[pd[8]][k]));
  }
/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/
  
  cdi = el->child[1]->dof[node0+2][n0];
  for (k = 0; k < DIM_OF_WORLD; k++)
    v[cdi][k] = (-0.125*v[pd[0]][k] + 0.375*v[pd[1]][k] + 0.75*v[pd[4]][k]);

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/
  
  for (i = 1; i < n; i++)
  {
    el = list[i].el_info.el;
    get_dof_indices(el, admin, pd);

    lr_set = 0;
    if (list[i].neigh[0]  &&  list[i].neigh[0]->no < i)
      lr_set = 1;

    if (list[i].neigh[1]  &&  list[i].neigh[1]->no < i)
      lr_set += 2;

    DEBUG_TEST_EXIT(lr_set, "no values set on both neighbours\n");

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

    switch (lr_set)
    {
    case 1:
      cdi = el->child[0]->dof[node0+4][n0];      
      for (k = 0; k < DIM_OF_WORLD; k++)
	v[cdi][k] = (0.125*(-v[pd[0]][k] - v[pd[1]][k]) + 0.25*v[pd[4]][k]
		     + 0.5*(v[pd[5]][k] + v[pd[7]][k]));
      break;
    case 2:
      cdi = el->child[0]->dof[node0+5][n0];      
      for (k = 0; k < DIM_OF_WORLD; k++)
	v[cdi][k] = (0.125*(-v[pd[0]][k] - v[pd[1]][k]) + 0.25*v[pd[4]][k]
		     + 0.5*(v[pd[6]][k] + v[pd[8]][k]));
    }
  }
  return;
}

static void real_d_coarse_inter2_3d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list,
				    int n)
{
  FUNCNAME("real_d_coarse_inter2_3d");
  EL        *el;
  REAL_D    *v = nil;
  int       cd, pd, k;
  const DOF_ADMIN *admin;
  MESH      *mesh = nil;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  GET_STRUCT(admin,drdv->fe_space);
  GET_STRUCT(mesh,drdv->fe_space);

  cd = el->child[0]->dof[mesh->node[VERTEX]+3][admin->n0_dof[VERTEX]]; 
  pd = el->dof[mesh->node[EDGE]][admin->n0_dof[EDGE]]; 
  for (k = 0; k < DIM_OF_WORLD; k++)
    v[pd][k] = v[cd][k];

  return;
}

static void real_d_coarse_restr2_3d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list,
				    int n)
{
  FUNCNAME("real_d_coarse_restr2_3d");
  EL        *el;
  REAL_D    *v = nil;
  const DOF *cd;
  DOF       pd[N_BAS2_3D], cdi;
  int       i, k, lr_set;
  int       node0, n0;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  get_dof_indices = drdv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drdv->fe_space);

  get_dof_indices(el, admin, pd);

  node0 = drdv->fe_space->mesh->node[EDGE];
  n0 = admin->n0_dof[EDGE];

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[0], admin, nil);

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[pd[0]][k] += (0.375*v[cd[6]][k] + 0.125*(-v[cd[8]][k] - v[cd[9]][k]));
    v[pd[1]][k] += (0.125*(-v[cd[6]][k] - v[cd[8]][k] - v[cd[9]][k]));
    v[pd[4]][k] = (v[cd[3]][k] + 0.75*v[cd[6]][k] 
		   + 0.25*(v[cd[8]][k] + v[cd[9]][k]));
    v[pd[5]][k] += (0.5*v[cd[8]][k]);
    v[pd[6]][k] += (0.5*v[cd[9]][k]);
    v[pd[7]][k] += (0.5*v[cd[8]][k]);
    v[pd[8]][k] += (0.5*v[cd[9]][k]);
  }
/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

  cd = get_dof_indices(el->child[1], admin, nil);
  cdi = el->child[1]->dof[node0+2][n0];

  for (k = 0; k < DIM_OF_WORLD; k++)
  {
    v[pd[0]][k] += (-0.125*v[cdi][k]);
    v[pd[1]][k] += (0.375*v[cdi][k]);
    v[pd[4]][k] += (0.75*v[cdi][k]);
  }

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/
  
  for (i = 1; i < n; i++)
  {
    el = list[i].el_info.el;
    get_dof_indices(el, admin, pd);

    lr_set = 0;
    if (list[i].neigh[0]  &&  list[i].neigh[0]->no < i)
      lr_set = 1;

    if (list[i].neigh[1]  &&  list[i].neigh[1]->no < i)
      lr_set += 2;

    DEBUG_TEST_EXIT(lr_set, "no values set on both neighbours\n");

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

    cd = get_dof_indices(el->child[0], admin, nil);

    switch(lr_set)
    {
    case 1:
      cdi = el->child[0]->dof[node0+4][n0];      
      for (k = 0; k < DIM_OF_WORLD; k++)
      {
	v[pd[0]][k] += (-0.125*v[cdi][k]);
	v[pd[1]][k] += (-0.125*v[cdi][k]);
	v[pd[4]][k] += (0.25*v[cdi][k]);
	v[pd[5]][k] += (0.5*v[cdi][k]);
	v[pd[7]][k] += (0.5*v[cdi][k]);
      }
      break;
    case 2:
      cdi = el->child[0]->dof[node0+5][n0];      
      for (k = 0; k < DIM_OF_WORLD; k++)
      {
	v[pd[0]][k] += (-0.125*v[cdi][k]);
	v[pd[1]][k] += (-0.125*v[cdi][k]);
	v[pd[4]][k] += (0.25*v[cdi][k]);
	v[pd[6]][k] += (0.5*v[cdi][k]);
	v[pd[8]][k] += (0.5*v[cdi][k]);
      }
      break;
    }
  }
  return;
}

static BAS_FCT      *phi2_3d[N_BAS2_3D]     = {phi2v0_3d, phi2v1_3d,
					       phi2v2_3d, phi2v3_3d,
					       phi2e0_3d, phi2e1_3d,
					       phi2e2_3d, phi2e3_3d,
					       phi2e4_3d, phi2e5_3d};
static GRD_BAS_FCT  *grd_phi2_3d[N_BAS2_3D] = {grd_phi2v0_3d, grd_phi2v1_3d,
					       grd_phi2v2_3d, grd_phi2v3_3d, 
					       grd_phi2e0_3d, grd_phi2e1_3d,
					       grd_phi2e2_3d, grd_phi2e3_3d,
					       grd_phi2e4_3d, grd_phi2e5_3d};
static D2_BAS_FCT   *D2_phi2_3d[N_BAS2_3D]  = {D2_phi2v0_3d, D2_phi2v1_3d,
					       D2_phi2v2_3d, D2_phi2v3_3d,
					       D2_phi2e0_3d, D2_phi2e1_3d,
					       D2_phi2e2_3d, D2_phi2e3_3d,
					       D2_phi2e4_3d, D2_phi2e5_3d};

static const BAS_FCTS  lagrange2_3d = {"lagrange2_3d", 3, N_BAS2_3D, 2,
				       {1,0,1,0},/* VERTEX, CENTER, EDGE,FACE*/
				       nil,
				       phi2_3d, grd_phi2_3d, D2_phi2_3d, 
				       get_dof_indices2_3d, 
				       get_bound2_3d, 
				       interpol2_3d,
				       interpol_d2_3d,
				       get_int_vec2_3d,
				       get_real_vec2_3d,
				       get_real_d_vec2_3d,
				       get_uchar_vec2_3d,
				       get_schar_vec2_3d,
				       real_refine_inter2_3d,
				       real_coarse_inter2_3d,
				       real_coarse_restr2_3d,
				       real_d_refine_inter2_3d,
				       real_d_coarse_inter2_3d,
				       real_d_coarse_restr2_3d,
				       bary2_3d,};

