/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "AbstractBrush.h"

#include <cmath>

#include <GTLCore/Color.h>
#include <GTLCore/Debug.h>

#include "DrawingPoint.h"
#include "DistanceInformation.h"

using namespace OpenRijn;

struct AbstractBrush::Private {
  bool supportDrawPoint;
  GTLCore::Color color;
};

AbstractBrush::AbstractBrush() : d(new Private)
{
  d->supportDrawPoint = true;
}

AbstractBrush::~AbstractBrush()
{
  delete d;
}

struct Vector {
  inline Vector(float _x, float _y) : x(_x), y(_y) {}
  float x, y;
  inline float norm() const {
    return sqrt(x*x + y*y);
  }
  inline void normalize() {
    float n = norm();
    x /= n;
    y /= n;
  }
};
inline Vector operator-(const Vector& v1, const Vector& v2)
{
  return Vector(v1.x - v2.x, v1.y - v2.y);
}
inline Vector operator+(const Vector& v1, const Vector& v2)
{
  return Vector(v1.x + v2.x, v1.y + v2.y);
}
inline Vector operator*(float v1, const Vector& v2)
{
  return Vector(v1 * v2.x, v1 * v2.y);
}
inline Vector operator*(const Vector& v1, float v2)
{
  return v2 * v1;
}

inline float angle(const Vector& v1, const Vector& v2)
{
  return acos( (v1.x * v2.x + v1.y * v2.y) / (v1.norm() * v2.norm()));
}

DrawingPoint mixDrawingPoint(Vector p, float t, const DrawingPoint& pt1, const DrawingPoint& pt2)
{
  return DrawingPoint(p.x, p.y, t * pt2.t + (1-t) * pt1.t, t * pt2.angle + (1-t) * pt1.angle);
}

DistanceInformation AbstractBrush::drawLine(AbstractCanvas* canvas, const DrawingPoint& pt1, const DrawingPoint& pt2, const DistanceInformation& information)
{
  Vector end(pt2.x, pt2.y);
  Vector start(pt1.x, pt1.y);

  Vector dragVec = end - start;

  GTL_ASSERT(information.distance >= 0);

  float endDist = dragVec.norm();
  float currentDist = information.distance;

    dragVec.normalize();
    Vector step(0, 0);

    float sp = information.spacing;
    while (currentDist < endDist) {

        Vector p = start +  currentDist * dragVec;

        float t = currentDist / endDist;

        sp = drawPoint(canvas, mixDrawingPoint(p, t, pt1, pt2));
        currentDist += sp;
    }

    return DistanceInformation(currentDist - endDist, sp);
}

#define BEZIER_FLATNESS_ANGLE 0.01

DistanceInformation AbstractBrush::drawCurve(AbstractCanvas* canvas, const OpenRijn::DrawingPoint& pt1, float x1, float y1, float x2, float y2, const OpenRijn::DrawingPoint& pt2, const OpenRijn::DistanceInformation& information)
{
    DistanceInformation newDistance;
    Vector control1(x1,y1);
    Vector control2(x2,y2);
    Vector pt1v(pt1.x, pt1.y);
    Vector pt2v(pt2.x, pt2.y);
    Vector d12 = pt2v - pt1v;
    
    if ((angle(d12, control1 - pt1v) < BEZIER_FLATNESS_ANGLE && angle(d12, control2 - pt2v) < BEZIER_FLATNESS_ANGLE) ) {
        newDistance = drawLine(canvas, pt1, pt2, information);
    } else {
        // Midpoint subdivision. See Foley & Van Dam Computer Graphics P.508
        Vector l2 = (pt1v + control1) * 0.5;
        Vector h = (control1 + control2) * 0.5;
        Vector l3 = (l2 + h) * 0.5;
        Vector r3 = (control2 + pt2v) * 0.5;
        Vector r2 = (h + r3) * 0.5;
        Vector l4 = (l3 + r2) * 0.5;

        DrawingPoint middlePI = mixDrawingPoint(l4, 0.5, pt1, pt2);
        
        newDistance = drawCurve(canvas, pt1, l2.x, l2.y, l3.x, l3.y, middlePI, information);
        newDistance = drawCurve(canvas, middlePI, r2.x, l2.y, r3.x, r3.y, pt2, newDistance);
    }

    return newDistance;
}

void AbstractBrush::setPaintColor(const GTLCore::Color& color)
{
  d->color = color;
}

GTLCore::Color AbstractBrush::paintColor() const
{
  return d->color;
}

bool AbstractBrush::supportDrawPoint() const
{
  return d->supportDrawPoint;
}

void AbstractBrush::setSupportDrawPoint(bool value)
{
  d->supportDrawPoint = value;
}
