/**
 *  Copyright 2007-2008 University Of Southern California
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


package edu.isi.pegasus.planner.common;


import edu.isi.pegasus.common.logging.LogManager;
import edu.isi.pegasus.planner.classes.PlannerOptions;

import java.util.Collection;
import java.util.Set;
import java.util.Vector;

/**
 * A Singleton wrapper around the
 * PlannerOptions class to get hold
 * of the options specified by the
 * user to run Pegasus. This allows
 * us to get hold of the options anywhere
 * without us messing up with the constructors
 * of classes.
 *
 * @author Karan Vahi
 * @author Gaurang Mehta
 * @version $Revision: 2572 $
 *
 * @see org.griphyn.cPlanner.classes.PlannerOptions
 */

public class UserOptions {
    /**
     * The object holding the
     * Singleton instance.
     */
    private static UserOptions mPOptions = null;

    /**
     * The Logger object used to
     * log the messages.
     */
    private LogManager mLogger;

    /**
     * The object containing all the
     * user options.
     */
    private PlannerOptions mPlannerOpts = null;

    /**
     * The method used to invoke the
     * singleton instance the first
     * time, from the CPlanner class.
     *
     * @param opt  the object containing
     *             the options to the Planner.
     */
    public static UserOptions getInstance(PlannerOptions opt){
        if(mPOptions == null && opt != null){
            mPOptions = new UserOptions(opt);
        }
        return mPOptions;
    }

    /**
     * The method used to invoke the
     * singleton instance from the
     * other classes.
     */
    public static UserOptions getInstance(){
        return getInstance(null);
    }


    /**
     * The private constructor that initializes the user options.
     * This should be invoked from the CPlanner class in the toolkit
     * package.
     *
     * @throws RuntimeException in case of incorrect initialization.
     */
    private UserOptions(PlannerOptions opt) {
        if(opt == null){
            mLogger.log("Wrong Invocation of the Singleton Instance",
                        LogManager.FATAL_MESSAGE_LEVEL);
            throw new RuntimeException( "Wrong Invocation of the Singleton Instance" );
        }
        mPlannerOpts = opt;
    }

    /**
     * Returns the dax file, that
     * was generated by Chimera
     * containing the description of
     * the Abstract plan.
     */
    public String getDaxFile(){
        return this.mPlannerOpts.getDAX();
    }

    /**
     * Returns output pool where the
     * user wants the data products
     * to be transferred to.
     */
    public String getOutputPool(){
        return this.mPlannerOpts.getOutputSite();
    }

    /**
     * Returns the options passed to the planner.
     *
     * @return PlannerOptions
     */
    public PlannerOptions getOptions(){
        return this.mPlannerOpts;
    }

    /**
     * Returns the directory where the
     * user wants the submit files to be
     * generated.
     */
    public String getSubmitFileDir(){
        return this.mPlannerOpts.getSubmitDirectory();
    }

    /**
     * Returns the list of execution pools
     * where the user wants the data
     * products to be computed.
     */
    public Collection getExecutionSites(){
        //return new Vector(this.mPlannerOpts.getExecutionSites());
        return this.mPlannerOpts.getExecutionSites();
    }

    /**
     * Sets the execution sites.
     *
     * @param sites  the <code>Set</code> containing the site names.
     */
    public void setExecutionSites(Collection sites){
        this.mPlannerOpts.setExecutionSites(sites);
    }

    /**
     * Returns whether we want to create
     * a Random Directory or not.
     */
    public boolean generateRandomDirectory(){
        return this.mPlannerOpts.generateRandomDirectory();
    }

    /**
     * It returns the set of cache files.
     */
    public Set getCacheFiles(){
        return this.mPlannerOpts.getCacheFiles();
    }
    /**
     * Returns the name of the random
     * directory, only if the generate
     * Random Dir flag is set.
     * Else it returns null.
     */
    public String getRandomDirName(){
        if ( this .generateRandomDirectory()){
            return this.mPlannerOpts.getRandomDir();
        }
        return null;
    }

    /**
     * Returns true if the user had asked
     * for the force option.
     */
    public boolean getForceOption(){
        return this.mPlannerOpts.getForce();
    }
}
