% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcmc-parcoord.R
\name{MCMC-parcoord}
\alias{MCMC-parcoord}
\alias{mcmc_parcoord}
\alias{mcmc_parcoord_data}
\alias{parcoord_style_np}
\title{Parallel coordinates plot of MCMC draws}
\usage{
mcmc_parcoord(
  x,
  pars = character(),
  regex_pars = character(),
  transformations = list(),
  ...,
  size = 0.2,
  alpha = 0.3,
  np = NULL,
  np_style = parcoord_style_np()
)

mcmc_parcoord_data(
  x,
  pars = character(),
  regex_pars = character(),
  transformations = list(),
  np = NULL
)

parcoord_style_np(div_color = "red", div_size = 0.2, div_alpha = 0.2)
}
\arguments{
\item{x}{A 3-D array, matrix, list of matrices, or data frame of MCMC draws.
The \link{MCMC-overview} page provides details on how to specify each these
allowed inputs. It is also possible to use an object with an
\code{as.array()} method that returns the same kind of 3-D array described on
the \link{MCMC-overview} page.}

\item{pars}{An optional character vector of parameter names. If neither
\code{pars} nor \code{regex_pars} is specified then the default is to use \emph{all}
parameters. As of version \verb{1.7.0}, \strong{bayesplot} also supports 'tidy'
parameter selection by specifying \code{pars = vars(...)}, where \code{...} is
specified the same way as in \link[dplyr:select]{dplyr::select(...)} and
similar functions. Examples of using \code{pars} in this way can be found on the
\link[=tidy-params]{Tidy parameter selection} page.}

\item{regex_pars}{An optional \link[base:grep]{regular expression} to use for
parameter selection. Can be specified instead of \code{pars} or in addition to
\code{pars}. When using \code{pars} for tidy parameter selection, the \code{regex_pars}
argument is ignored since \link[tidyselect:select_helpers]{select helpers}
perform a similar function.}

\item{transformations}{Optionally, transformations to apply to parameters
before plotting. If \code{transformations} is a function or a single string
naming a function then that function will be used to transform all
parameters. To apply transformations to particular parameters, the
\code{transformations} argument can be a named list with length equal to
the number of parameters to be transformed. Currently only univariate
transformations of scalar parameters can be specified (multivariate
transformations will be implemented in a future release). If
\code{transformations} is a list, the name of each list element should be a
parameter name and the content of each list element should be a function
(or any item to match as a function via \code{\link[=match.fun]{match.fun()}}, e.g. a
string naming a function). If a function is specified by its name as a
string (e.g. \code{"log"}), then it can be used to construct a new
parameter label for the appropriate parameter (e.g. \code{"log(sigma)"}).
If a function itself is specified
(e.g. \code{log} or \code{function(x) log(x)})
then \code{"t"} is used in the new parameter label to indicate that the
parameter is transformed (e.g. \code{"t(sigma)"}).

Note: due to partial argument matching \code{transformations} can be
abbreviated for convenience in interactive use (e.g., \code{transform}).}

\item{...}{Currently ignored.}

\item{size, alpha}{Arguments passed on to \code{\link[ggplot2:geom_line]{ggplot2::geom_line()}}.}

\item{np}{For models fit using \link{NUTS} (more generally,
any \href{https://en.wikipedia.org/wiki/Symplectic_integrator}{symplectic integrator}),
an optional data frame providing NUTS diagnostic information. The data
frame should be the object returned by \code{\link[=nuts_params]{nuts_params()}} or one with the same
structure.}

\item{np_style}{A call to the \code{parcoord_style_np()} helper function to
specify arguments controlling the appearance of superimposed lines
representing NUTS diagnostics (in this case divergences) if the \code{np}
argument is specified.}

\item{div_color, div_size, div_alpha}{Optional arguments to the
\code{parcoord_style_np()} helper function that are eventually passed to
\code{\link[ggplot2:geom_line]{ggplot2::geom_line()}} if the \code{np} argument is also specified. They control
the color, size, and transparency specifications for showing divergences in
the plot. The default values are displayed in the \strong{Usage} section above.}
}
\value{
The plotting functions return a ggplot object that can be further
customized using the \strong{ggplot2} package. The functions with suffix
\verb{_data()} return the data that would have been drawn by the plotting
function.
}
\description{
Parallel coordinates plot of MCMC draws (one dimension per parameter).
See the \strong{Plot Descriptions} section below for details,
and see \href{https://github.com/jgabry/bayes-vis-paper#readme}{Gabry et al. (2019)}
for more background and a real example.
}
\section{Plot Descriptions}{

\describe{
\item{\code{mcmc_parcoord()}}{
\href{https://en.wikipedia.org/wiki/Parallel_coordinates}{Parallel coordinates plot}
of MCMC draws. There is one dimension per parameter along the horizontal
axis and each set of connected line segments represents a single MCMC draw
(i.e., a vector of length equal to the number of parameters).

The parallel coordinates plot is most useful if the optional HMC/NUTS
diagnostic information is provided via the \code{np} argument. In that
case divergences are highlighted in the plot. The appearance of the
divergences can be customized using the \code{np_style} argument and the
\code{parcoord_style_np} helper function. This version of the plot is the
same as the parallel coordinates plot described in Gabry et al. (2019).

When the plotted model parameters are on very different scales the
\code{transformations} argument can be useful. For example, to standardize
all variables before plotting you could use function \code{(x - mean(x))/sd(x)}
when specifying the \code{transformations} argument to
\code{mcmc_parcoord}. See the \strong{Examples} section for how to do this.
}
}
}

\examples{
color_scheme_set("pink")
x <- example_mcmc_draws(params = 5)
mcmc_parcoord(x)
mcmc_parcoord(x, regex_pars = "beta")

\dontrun{
# Example using a Stan demo model
library(rstan)
fit <- stan_demo("eight_schools")
draws <- as.array(fit, pars = c("mu", "tau", "theta", "lp__"))
np <- nuts_params(fit)
str(np)
levels(np$Parameter)

color_scheme_set("brightblue")
mcmc_parcoord(draws, alpha = 0.05)
mcmc_parcoord(draws, np = np)

# customize appearance of divergences
color_scheme_set("darkgray")
div_style <- parcoord_style_np(div_color = "green", div_size = 0.05, div_alpha = 0.4)
mcmc_parcoord(draws, size = 0.25, alpha = 0.1,
              np = np, np_style = div_style)

# to use a transformation (e.g., standardizing all the variables can be helpful)
# specify the 'transformations' argument (though partial argument name
# matching means we can just use 'trans' or 'transform')
mcmc_parcoord(
  draws,
  transform = function(x) {(x - mean(x)) / sd(x)},
  size = 0.25,
  alpha = 0.1,
  np = np,
  np_style = div_style
 )

# mcmc_parcoord_data returns just the data in a conventient form for plotting
d <- mcmc_parcoord_data(x, np = np)
head(d)
tail(d)
}

}
\references{
Gabry, J. , Simpson, D. , Vehtari, A. , Betancourt, M. and
Gelman, A. (2019), Visualization in Bayesian workflow.
\emph{J. R. Stat. Soc. A}, 182: 389-402. doi:10.1111/rssa.12378.
(\href{https://rss.onlinelibrary.wiley.com/doi/full/10.1111/rssa.12378}{journal version},
\href{https://arxiv.org/abs/1709.01449}{arXiv preprint},
\href{https://github.com/jgabry/bayes-vis-paper}{code on GitHub})

Hartikainen, A. (2017, Aug 23). Concentration of divergences
(Msg 21). Message posted to The Stan Forums:
\url{https://discourse.mc-stan.org/t/concentration-of-divergences/1590/21}.
}
\seealso{
Other MCMC: 
\code{\link{MCMC-combos}},
\code{\link{MCMC-diagnostics}},
\code{\link{MCMC-distributions}},
\code{\link{MCMC-intervals}},
\code{\link{MCMC-nuts}},
\code{\link{MCMC-overview}},
\code{\link{MCMC-recover}},
\code{\link{MCMC-scatterplots}},
\code{\link{MCMC-traces}}
}
\concept{MCMC}
