% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods_dbscan.R, R/methods_hclust.R,
%   R/methods_kmeans.R, R/methods_mclust.R, R/methods_pam.R
\name{model_parameters.dbscan}
\alias{model_parameters.dbscan}
\alias{model_parameters.hclust}
\alias{model_parameters.pvclust}
\alias{model_parameters.kmeans}
\alias{model_parameters.hkmeans}
\alias{model_parameters.Mclust}
\alias{model_parameters.pam}
\title{Parameters from Cluster Models (k-means, ...)}
\usage{
\method{model_parameters}{dbscan}(model, data = NULL, clusters = NULL, ...)

\method{model_parameters}{hclust}(model, data = NULL, clusters = NULL, ...)

\method{model_parameters}{pvclust}(model, data = NULL, clusters = NULL, ci = 0.95, ...)

\method{model_parameters}{kmeans}(model, ...)

\method{model_parameters}{hkmeans}(model, ...)

\method{model_parameters}{Mclust}(model, data = NULL, clusters = NULL, ...)

\method{model_parameters}{pam}(model, data = NULL, clusters = NULL, ...)
}
\arguments{
\item{model}{Cluster model.}

\item{data}{A data.frame.}

\item{clusters}{A vector with clusters assignments (must be same length as rows in data).}

\item{...}{Arguments passed to or from other methods.}

\item{ci}{Confidence Interval (CI) level. Default to \code{0.95} (\verb{95\%}).}
}
\description{
Format cluster models obtained for example by \code{\link[=kmeans]{kmeans()}}.
}
\examples{
\donttest{
# DBSCAN ---------------------------
if (require("dbscan", quietly = TRUE)) {
  model <- dbscan::dbscan(iris[1:4], eps = 1.45, minPts = 10)

  rez <- model_parameters(model, iris[1:4])
  rez

  # Get clusters
  predict(rez)

  # Clusters centers in long form
  attributes(rez)$means

  # Between and Total Sum of Squares
  attributes(rez)$Sum_Squares_Total
  attributes(rez)$Sum_Squares_Between

  # HDBSCAN
  model <- dbscan::hdbscan(iris[1:4], minPts = 10)
  model_parameters(model, iris[1:4])
}
}
#
# Hierarchical clustering (hclust) ---------------------------
data <- iris[1:4]
model <- hclust(dist(data))
clusters <- cutree(model, 3)

rez <- model_parameters(model, data, clusters)
rez

# Get clusters
predict(rez)

# Clusters centers in long form
attributes(rez)$means

# Between and Total Sum of Squares
attributes(rez)$Total_Sum_Squares
attributes(rez)$Between_Sum_Squares
\donttest{
#
# pvclust (finds "significant" clusters) ---------------------------
if (require("pvclust", quietly = TRUE)) {
  data <- iris[1:4]
  # NOTE: pvclust works on transposed data
  model <- pvclust::pvclust(datawizard::data_transpose(data),
    method.dist = "euclidean",
    nboot = 50,
    quiet = TRUE
  )

  rez <- model_parameters(model, data, ci = 0.90)
  rez

  # Get clusters
  predict(rez)

  # Clusters centers in long form
  attributes(rez)$means

  # Between and Total Sum of Squares
  attributes(rez)$Sum_Squares_Total
  attributes(rez)$Sum_Squares_Between
}
}
\dontrun{
#
# K-means -------------------------------
model <- kmeans(iris[1:4], centers = 3)
rez <- model_parameters(model)
rez

# Get clusters
predict(rez)

# Clusters centers in long form
attributes(rez)$means

# Between and Total Sum of Squares
attributes(rez)$Sum_Squares_Total
attributes(rez)$Sum_Squares_Between
}
\dontrun{
#
# Hierarchical K-means (factoextra::hkclust) ----------------------
if (require("factoextra", quietly = TRUE)) {
  data <- iris[1:4]
  model <- factoextra::hkmeans(data, k = 3)

  rez <- model_parameters(model)
  rez

  # Get clusters
  predict(rez)

  # Clusters centers in long form
  attributes(rez)$means

  # Between and Total Sum of Squares
  attributes(rez)$Sum_Squares_Total
  attributes(rez)$Sum_Squares_Between
}
}
if (require("mclust", quietly = TRUE)) {
  model <- mclust::Mclust(iris[1:4], verbose = FALSE)
  model_parameters(model)
}
\dontrun{
#
# K-Medoids (PAM and HPAM) ==============
if (require("cluster", quietly = TRUE)) {
  model <- cluster::pam(iris[1:4], k = 3)
  model_parameters(model)
}
if (require("fpc", quietly = TRUE)) {
  model <- fpc::pamk(iris[1:4], criterion = "ch")
  model_parameters(model)
}
}
}
