-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem)
procedure Check_Interrupt_Property_Consistency
  (Has_Interrupt_Property : in     Boolean;
   Sym                    : in     Dictionary.Symbol;
   Type_Sym               : in     Dictionary.Symbol;
   Error_Node_Pos         : in     LexTokenManager.Token_Position;
   Consistent             :    out Boolean)
is

   function Type_Can_Interrupt (Type_Sym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      -- To interrupt the type must be a protected type with an attach handler.
      return Dictionary.IsProtectedTypeMark (Type_Sym)
        and then Dictionary.GetTypeHasPragma (Dictionary.GetRootType (Type_Sym), Dictionary.AttachHandler);
   end Type_Can_Interrupt;

begin -- Check_Interrupt_Property_Consistency
   Consistent := True;
   if Dictionary.Is_Declared (Item => Type_Sym) or else Dictionary.IsPredefined (Type_Sym) then
      if Has_Interrupt_Property and then not Type_Can_Interrupt (Type_Sym => Type_Sym) then
         -- We have an own variable with the interrupt property whose type
         -- cannot interrupt.
         Consistent := False;
         ErrorHandler.Semantic_Error
           (Err_Num   => 936,
            Reference => ErrorHandler.No_Reference,
            Position  => Error_Node_Pos,
            Id_Str    => Dictionary.GetSimpleName (Sym));
      elsif not Has_Interrupt_Property and then Type_Can_Interrupt (Type_Sym => Type_Sym) then
         -- We have an own variable without the interrupt property whose
         -- type can interrupt.
         Consistent := False;
         ErrorHandler.Semantic_Error
           (Err_Num   => 935,
            Reference => ErrorHandler.No_Reference,
            Position  => Error_Node_Pos,
            Id_Str    => Dictionary.GetSimpleName (Sym));
      end if;
   end if;
end Check_Interrupt_Property_Consistency;
