#![cfg(dh)]

use wolfssl::wolfcrypt::dh::DH;
use wolfssl::wolfcrypt::random::RNG;

#[test]
#[cfg(dh_keygen)]
fn test_dh_named_parameters() {
    assert_eq!(DH::get_min_key_size_for_named_parameters(DH::FFDHE_2048), 29);

    let mut p_size = 0u32;
    let mut g_size = 0u32;
    let mut q_size = 0u32;
    DH::get_named_parameter_sizes(DH::FFDHE_2048, &mut p_size, &mut g_size, &mut q_size);

    assert!(p_size > 0u32);
    assert!(g_size > 0u32);
    let mut dh = DH::new_named(DH::FFDHE_2048).expect("Error with new_named()");

    let mut p = [0u8; 256];
    let mut q = [0u8; 256];
    let mut g = [0u8; 256];
    dh.export_params_raw(&mut p, &mut p_size, &mut q, &mut q_size, &mut g, &mut g_size).expect("Error with export_params_raw()");

    assert_ne!(p, [0u8; 256]);
    assert_ne!(g, [0u8; 256]);
}

#[test]
#[cfg(dh_keygen)]
fn test_generate_params() {
    let mut rng = RNG::new().expect("Error with RNG::new()");
    let mut dh = DH::generate(&mut rng, 2048).expect("Error with generate()");

    let mut private = [0u8; 256];
    let mut private_size = 0u32;
    let mut public = [0u8; 256];
    let mut public_size = 0u32;
    dh.generate_key_pair(&mut rng, &mut private, &mut private_size, &mut public, &mut public_size).expect("Error with generate_key_pair()");
}

#[test]
#[cfg(dh_keygen)]
fn test_generate_key_pair() {
    let mut rng = RNG::new().expect("Error with RNG::new()");
    let mut dh = DH::new_named(DH::FFDHE_2048).expect("Error with new_named()");

    let mut p = [0u8; 256];
    let mut q = [0u8; 256];
    let mut g = [0u8; 256];
    let mut p_size = 0u32;
    let mut q_size = 0u32;
    let mut g_size = 0u32;
    dh.export_params_raw(&mut p, &mut p_size, &mut q, &mut q_size, &mut g, &mut g_size).expect("Error with export_params_raw()");
    let p = &p[0..(p_size as usize)];
    let g = &g[0..(g_size as usize)];

    assert!(DH::compare_named_key(DH::FFDHE_2048, p, g, None));

    let mut private = [0u8; 256];
    let mut private_size = 0u32;
    let mut public = [0u8; 256];
    let mut public_size = 0u32;
    dh.generate_key_pair(&mut rng, &mut private, &mut private_size, &mut public, &mut public_size).expect("Error with generate_key_pair()");

    let private = &private[0..(private_size as usize)];
    let public = &public[0..(public_size as usize)];
    dh.check_key_pair(public, private).expect("Error with check_key_pair()");
    dh.check_priv_key(private).expect("Error with check_priv_key()");
    dh.check_pub_key(public).expect("Error with check_pub_key()");
}

#[test]
fn test_dh_checks() {
    let p = [
        0xc5u8, 0x7c, 0xa2, 0x4f, 0x4b, 0xd6, 0x8c, 0x3c,
        0xda, 0xc7, 0xba, 0xaa, 0xea, 0x2e, 0x5c, 0x1e,
        0x18, 0xb2, 0x7b, 0x8c, 0x55, 0x65, 0x9f, 0xea,
        0xe0, 0xa1, 0x36, 0x53, 0x2b, 0x36, 0xe0, 0x4e,
        0x3e, 0x64, 0xa9, 0xe4, 0xfc, 0x8f, 0x32, 0x62,
        0x97, 0xe4, 0xbe, 0xf7, 0xc1, 0xde, 0x07, 0x5a,
        0x89, 0x28, 0xf3, 0xfe, 0x4f, 0xfe, 0x68, 0xbc,
        0xfb, 0x0a, 0x7c, 0xa4, 0xb3, 0x14, 0x48, 0x89,
        0x9f, 0xaf, 0xb8, 0x43, 0xe2, 0xa0, 0x62, 0x5c,
        0xb4, 0x88, 0x3f, 0x06, 0x50, 0x11, 0xfe, 0x65,
        0x8d, 0x49, 0xd2, 0xf5, 0x4b, 0x74, 0x79, 0xdb,
        0x06, 0x62, 0x92, 0x89, 0xed, 0xda, 0xcb, 0x87,
        0x37, 0x16, 0xd2, 0xa1, 0x7a, 0xe8, 0xde, 0x92,
        0xee, 0x3e, 0x41, 0x4a, 0x91, 0x5e, 0xed, 0xf3,
        0x6c, 0x6b, 0x7e, 0xfd, 0x15, 0x92, 0x18, 0xfc,
        0xa7, 0xac, 0x42, 0x85, 0x57, 0xe9, 0xdc, 0xda,
        0x55, 0xc9, 0x8b, 0x28, 0x9e, 0xc1, 0xc4, 0x46,
        0x4d, 0x88, 0xed, 0x62, 0x8e, 0xdb, 0x3f, 0xb9,
        0xd7, 0xc8, 0xe3, 0xcf, 0xb8, 0x34, 0x2c, 0xd2,
        0x6f, 0x28, 0x06, 0x41, 0xe3, 0x66, 0x8c, 0xfc,
        0x72, 0xff, 0x26, 0x3b, 0x6b, 0x6c, 0x6f, 0x73,
        0xde, 0xf2, 0x90, 0x29, 0xe0, 0x61, 0x32, 0xc4,
        0x12, 0x74, 0x09, 0x52, 0xec, 0xf3, 0x1b, 0xa6,
        0x45, 0x98, 0xac, 0xf9, 0x1c, 0x65, 0x8e, 0x3a,
        0x91, 0x84, 0x4b, 0x23, 0x8a, 0xb2, 0x3c, 0xc9,
        0xfa, 0xea, 0xf1, 0x38, 0xce, 0xd8, 0x05, 0xe0,
        0xfa, 0x44, 0x68, 0x1f, 0xeb, 0xd9, 0x57, 0xb8,
        0x4a, 0x97, 0x5b, 0x88, 0xc5, 0xf1, 0xbb, 0xb0,
        0x49, 0xc3, 0x91, 0x7c, 0xd3, 0x13, 0xb9, 0x47,
        0xbb, 0x91, 0x8f, 0xe5, 0x26, 0x07, 0xab, 0xa9,
        0xc5, 0xd0, 0x3d, 0x95, 0x41, 0x26, 0x92, 0x9d,
        0x13, 0x67, 0xf2, 0x7e, 0x11, 0x88, 0xdc, 0x2d
    ];
    let g = [
        0x4au8, 0x1a, 0xf3, 0xa4, 0x92, 0xe9, 0xee, 0x74,
        0x6e, 0x57, 0xd5, 0x8c, 0x2c, 0x5b, 0x41, 0x41,
        0x5e, 0xd4, 0x55, 0x19, 0xdc, 0xd9, 0x32, 0x91,
        0xf7, 0xfd, 0xc2, 0x57, 0xff, 0x03, 0x14, 0xdb,
        0xf1, 0xb7, 0x60, 0x0c, 0x43, 0x59, 0x3f, 0xff,
        0xac, 0xf1, 0x80, 0x9a, 0x15, 0x6f, 0xd8, 0x6e,
        0xb7, 0x85, 0x18, 0xc8, 0xec, 0x4e, 0x59, 0x4a,
        0xe2, 0x91, 0x43, 0x4c, 0xeb, 0x95, 0xb6, 0x2e,
        0x9a, 0xea, 0x53, 0x68, 0x80, 0x64, 0x69, 0x40,
        0xf9, 0xec, 0xbd, 0x85, 0x89, 0x26, 0x97, 0x67,
        0xaf, 0xb0, 0xad, 0x00, 0x1b, 0xd4, 0xfd, 0x94,
        0xd3, 0xe9, 0x92, 0xb1, 0xb4, 0xbc, 0x5a, 0xaa,
        0x92, 0x80, 0x89, 0x3b, 0x39, 0x05, 0x6c, 0x22,
        0x26, 0xfe, 0x5a, 0x28, 0x6c, 0x37, 0x50, 0x5a,
        0x38, 0x99, 0xcf, 0xf3, 0xc1, 0x96, 0x45, 0xdc,
        0x01, 0xcb, 0x20, 0x87, 0xa5, 0x00, 0x8c, 0xf5,
        0x4d, 0xc2, 0xef, 0xb8, 0x9b, 0xd1, 0x87, 0xbe,
        0xed, 0xd5, 0x0a, 0x29, 0x15, 0x34, 0x59, 0x4c,
        0x3a, 0x05, 0x22, 0x05, 0x44, 0x4f, 0x9f, 0xc8,
        0x47, 0x12, 0x24, 0x8e, 0xa8, 0x79, 0xe4, 0x67,
        0xba, 0x4d, 0x5b, 0x75, 0x56, 0x95, 0xeb, 0xe8,
        0x8a, 0xfa, 0x8e, 0x01, 0x8c, 0x1b, 0x74, 0x63,
        0xd9, 0x2f, 0xf7, 0xd3, 0x44, 0x8f, 0xa8, 0xf5,
        0xaf, 0x6c, 0x4f, 0xdb, 0xe7, 0xc9, 0x6c, 0x71,
        0x22, 0xa3, 0x1d, 0xf1, 0x40, 0xb2, 0xe0, 0x9a,
        0xb6, 0x72, 0xc9, 0xc0, 0x13, 0x16, 0xa2, 0x4a,
        0xe1, 0x92, 0xc7, 0x54, 0x23, 0xab, 0x9d, 0xa1,
        0xa1, 0xe5, 0x0b, 0xed, 0xba, 0xe8, 0x84, 0x37,
        0xb2, 0xe7, 0xfe, 0x32, 0x8d, 0xfa, 0x1c, 0x53,
        0x77, 0x97, 0xc7, 0xf3, 0x48, 0xc9, 0xdb, 0x2d,
        0x75, 0x52, 0x9d, 0x42, 0x51, 0x78, 0x62, 0x68,
        0x05, 0x45, 0x15, 0xf8, 0xa2, 0x4e, 0xf3, 0x0b
    ];
    let q = [
        0xe0u8, 0x35, 0x37, 0xaf, 0xb2, 0x50, 0x91, 0x8e,
        0xf2, 0x62, 0x2b, 0xd9, 0x9f, 0x6c, 0x11, 0x75,
        0xec, 0x24, 0x5d, 0x78, 0x59, 0xe7, 0x8d, 0xb5,
        0x40, 0x52, 0xed, 0x41
    ];
    let q0 = [
                                                  0x00u8,
        0xe0, 0x35, 0x37, 0xaf, 0xb2, 0x50, 0x91, 0x8e,
        0xf2, 0x62, 0x2b, 0xd9, 0x9f, 0x6c, 0x11, 0x75,
        0xec, 0x24, 0x5d, 0x78, 0x59, 0xe7, 0x8d, 0xb5,
        0x40, 0x52, 0xed, 0x41
    ];
    let mut rng = RNG::new().expect("Error with RNG::new()");
    let _dh = DH::new_from_pg(&p, &g).expect("Error with new_from_pg()");
    let _dh = DH::new_from_pgq(&p, &g, &q).expect("Error with new_from_pgq()");
    let mut dh = DH::new_from_pgq_with_check(&p, &g, &q, 0, &mut rng).expect("Error with new_from_pgq()");
    let mut private = [0u8; 256];
    let mut private_size = 0u32;
    let mut public = [0u8; 256];
    let mut public_size = 0u32;
    dh.generate_key_pair(&mut rng, &mut private, &mut private_size, &mut public, &mut public_size).expect("Error with generate_key_pair()");

    let private = &private[0..(private_size as usize)];
    let public = &public[0..(public_size as usize)];
    dh.check_priv_key_ex(private, Some(&q)).expect("Error with check_priv_key_ex()");
    DH::check_pub_value(&p, public).expect("Error with check_pub_value()");
    dh.check_pub_key_ex(public, &q0).expect("Error with check_pub_key_ex()");
}

#[test]
fn test_dh_shared_secret() {
    let mut rng = RNG::new().expect("Error with RNG::new()");
    let mut dh = DH::new_named(DH::FFDHE_2048).expect("Error with new_named()");

    let mut private0 = [0u8; 256];
    let mut private0_size = 0u32;
    let mut public0 = [0u8; 256];
    let mut public0_size = 0u32;
    dh.generate_key_pair(&mut rng, &mut private0, &mut private0_size, &mut public0, &mut public0_size).expect("Error with generate_key_pair()");
    let private0 = &private0[0..(private0_size as usize)];
    let public0 = &public0[0..(public0_size as usize)];

    let mut private1 = [0u8; 256];
    let mut private1_size = 0u32;
    let mut public1 = [0u8; 256];
    let mut public1_size = 0u32;
    dh.generate_key_pair(&mut rng, &mut private1, &mut private1_size, &mut public1, &mut public1_size).expect("Error with generate_key_pair()");
    let private1 = &private1[0..(private1_size as usize)];
    let public1 = &public1[0..(public1_size as usize)];

    let mut ss0 = [0u8; 256];
    let ss0_size = dh.shared_secret(&mut ss0, private0, public1).expect("Error with shared_secret()");
    let ss0 = &ss0[0..ss0_size];

    let mut ss1 = [0u8; 256];
    let ss1_size = dh.shared_secret(&mut ss1, private1, public0).expect("Error with shared_secret()");
    let ss1 = &ss1[0..ss1_size];

    assert_eq!(ss0_size, ss1_size);
    assert_eq!(*ss0, *ss1);
}
