# -*- coding: UTF-8 -*-

"""
Copyright (C) 2005-07 Adolfo González Blázquez <code@infinicode.org>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version. 

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details. 

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

If you find any bugs or have any suggestions email: code@infinicode.org
"""

import pygtk
pygtk.require("2.0")
import gtk
import gconf

from cameramonitor import camglobals as glob
from cameramonitor import autostart

class CameraMonitorPreferences:

	# Class init
	def __init__(self, state):

		self.state = state

		self.video_device = '/dev/video0'
		self.timeout = 5    #seconds
		self.notify = True
		self.autostart = False

		self.gconf_path = '/apps/cameramonitor'
		self.gconf_key_video_device = self.gconf_path + '/video_device'
		self.gconf_key_timeout = self.gconf_path + '/timeout'
		self.gconf_key_notify = self.gconf_path + '/notify'
		self.gconf_key_autostart = self.gconf_path + '/autostart'
			
		self.preferences_read()
		self.preferences_monitor_gconf()

		self.prefs = self.preferences_create()

	# Returns video device value
	def preferences_get_video_device(self):
		return self.video_device

	# Returns video device value
	def preferences_get_timeout(self):
		return self.timeout

	# Returns notify enabled?    
	def preferences_get_notify(self):
		return self.notify
		
	# Returns autostart enabled?    
	def preferences_get_autostart(self):
		return self.autostart

	# Save video device
	def preferences_save_video_device(self):
		client = gconf.client_get_default()
		client.set_string(self.gconf_key_video_device, self.video_device)

	# Save timeout
	def preferences_save_timeout(self):
		client = gconf.client_get_default()
		client.set_int(self.gconf_key_timeout, self.timeout)

	# Save notification preferences
	def preferences_save_notify(self):
		client = gconf.client_get_default()
		client.set_bool(self.gconf_key_notify, self.notify)

	# Save autostart preferences
	def preferences_save_autostart(self):
		client = gconf.client_get_default()
		client.set_bool(self.gconf_key_autostart, self.autostart)

	# Save preferences using GConf
	def preferences_save(self):
		self.preferences_save_video_device()
		self.preferences_save_timeout()
		self.preferences_save_notify()
		self.preferences_save_autostart()

	# Read preferences using GConf
	def preferences_read(self):
		client = gconf.client_get_default()
		video = client.get_string(self.gconf_key_video_device)
		timeout = client.get_int(self.gconf_key_timeout)
		notify = client.get_bool(self.gconf_key_notify)
		autostart = client.get_bool(self.gconf_key_autostart)

		if video != None: self.video_device = video
		else: client.set_string(self.gconf_key_video_device, self.video_device)
		if timeout > 0: self.timeout = timeout
		else: client.set_int(self.gconf_key_timeout, self.timeout)
		if notify != None: self.notify = notify
		else: client.set_bool(self.gconf_key_notify, self.notify)
		if autostart != None: self.autostart = autostart
		else: client.set_bool(self.gconf_key_autostart, self.autostart)        

	# Video device changed
	def device_changed(self, widget):
		self.video_device = widget.get_text()
						
	# Video device change finished
	def device_change_end(self, widget, event):
		self.preferences_save_video_device()
		
	# Timeout changed
	def timeout_changed(self, widget):
		self.timeout = widget.get_value_as_int()
		self.preferences_save_timeout()
		
	# Notify changed
	def notify_changed(self, widget):
		self.notify = widget.get_active()
		self.preferences_save_notify()

	# Autostart changed
	def autostart_changed(self, widget):
		self.autostart = widget.get_active()
		self.preferences_save_autostart()
		if self.autostart: autostart.create_autostart_file()
		else: autostart.delete_autostart_file()
		
	# This monitors gconf keys
	def preferences_monitor_gconf(self):
		client = gconf.client_get_default()
		client.add_dir(self.gconf_path, gconf.CLIENT_PRELOAD_NONE)
		client.notify_add(self.gconf_key_video_device, self.preferences_video_key_changed);
		client.notify_add(self.gconf_key_timeout, self.preferences_timeout_key_changed);
		client.notify_add(self.gconf_key_notify, self.preferences_notify_key_changed);
		client.notify_add(self.gconf_key_autostart, self.preferences_autostart_key_changed);

	# Callback: someone's changed the gconf key for video_device
	def preferences_video_key_changed(self, client, connection_id, entry, args):
		self.video_device = entry.get_value().get_string()
		self.dev_entry.set_text(self.video_device)

	# Callback: someone's changed the gconf key for video_device
	def preferences_timeout_key_changed(self, client, connection_id, entry, args):
		self.timeout = entry.get_value().get_int()
		self.tim_spin.set_value(self.timeout)

	# Callback: someone's changed the gconf key for notification
	def preferences_notify_key_changed(self, client, connection_id, entry, args):
		self.notify = entry.get_value().get_bool()
		self.notify_check.set_active(self.notify)

	# Callback: someone's changed the gconf key for autostart
	def preferences_autostart_key_changed(self, client, connection_id, entry, args):
		self.autostart = entry.get_value().get_bool()
		self.autostart_check.set_active(self.autostart)

	# Callback: display preferences dialog
	def preferences_show(self, widget):
		if self.prefs.get_children() == []: self.prefs = self.preferences_create()
		self.dev_entry.set_text(self.video_device)
		self.tim_spin.set_value(self.timeout)
		self.notify_check.set_active(self.notify)
		self.autostart_check.set_active(self.autostart)
		self.prefs.show_all()
	

	# Callback: button close pressed on preferences
	def preferences_close_clicked(self, widget):
		self.preferences_save()
		self.prefs.destroy()
		if self.state == 'alone': gtk.main_quit()

	# Creates the preferences dialog
	def preferences_create(self):

		prefs = gtk.Window()
		prefs.set_title("Camera Monitor Preferences")
		prefs.set_icon_from_file(glob.image)
		prefs.set_position(gtk.WIN_POS_CENTER_ALWAYS)
		prefs.set_resizable(False)
		prefs.connect("destroy", self.preferences_close_clicked)

		# Video Device Prefs
		dev_desc = gtk.Label('<b>Webcam device path</b>')
		dev_desc.set_use_markup(True)
		dev_desc.set_alignment(xalign=0, yalign=0.5)

		dev_label = gtk.Label("Video device:")
		dev_label.set_alignment(xalign=0, yalign=0.5)
		self.dev_entry = gtk.Entry()
		self.dev_entry.set_text(self.video_device)
		self.dev_entry.set_activates_default(True)
		self.dev_entry.connect('changed',self.device_changed)
		self.dev_entry.connect('focus-out-event',self.device_change_end)

		dev_hbox = gtk.HBox()
		dev_hbox.set_spacing(10)
		dev_hbox.pack_start(dev_label)
		dev_hbox.pack_start(self.dev_entry)

		dev_vbox = gtk.VBox()
		dev_vbox.set_border_width(10)
		dev_vbox.set_spacing(10)
		dev_vbox.pack_start(dev_desc)
		dev_vbox.pack_start(dev_hbox)

		# Timeout Prefs
		tim_desc = gtk.Label('<b>Select time between checks</b>')
		tim_desc.set_use_markup(True)
		tim_desc.set_alignment(xalign=0, yalign=0.5)

		tim_label = gtk.Label("Timeout (seconds):")
		tim_label.set_alignment(xalign=0, yalign=0.5)
		tim_adj = gtk.Adjustment(self.timeout, 1.0, 60.0, 1.0, 5.0, 0.0)
		self.tim_spin = gtk.SpinButton(tim_adj, 0, 0)
		self.tim_spin.set_activates_default(True)
		self.tim_spin.connect('changed',self.timeout_changed)

		tim_hbox = gtk.HBox()
		tim_hbox.set_spacing(10)
		tim_hbox.pack_start(tim_label)
		tim_hbox.pack_start(self.tim_spin)

		tim_vbox = gtk.VBox()
		tim_vbox.set_border_width(10)
		tim_vbox.set_spacing(10)
		tim_vbox.pack_start(tim_desc)
		tim_vbox.pack_start(tim_hbox)

		# Notification Prefs
		notify_desc = gtk.Label('<b>Notification</b>')
		notify_desc.set_use_markup(True)
		notify_desc.set_alignment(xalign=0, yalign=0.5)

		self.notify_check = gtk.CheckButton(label="Show notification when camera is switched")
		self.notify_check.set_alignment(xalign=0, yalign=0.5)
		self.notify_check.connect('toggled',self.notify_changed)

		notify_vbox = gtk.VBox()
		notify_vbox.set_border_width(10)
		notify_vbox.set_spacing(10)
		notify_vbox.pack_start(notify_desc)
		notify_vbox.pack_start(self.notify_check)

		# Autostart Prefs
		autostart_desc = gtk.Label('<b>Autostart</b>')
		autostart_desc.set_use_markup(True)
		autostart_desc.set_alignment(xalign=0, yalign=0.5)

		self.autostart_check = gtk.CheckButton(label="Autostart Camera Monitor with every session")
		self.autostart_check.set_alignment(xalign=0, yalign=0.5)
		self.autostart_check.connect('toggled',self.autostart_changed)

		autostart_vbox = gtk.VBox()
		autostart_vbox.set_border_width(10)
		autostart_vbox.set_spacing(10)
		autostart_vbox.pack_start(autostart_desc)
		autostart_vbox.pack_start(self.autostart_check)

		# OK and Cancel buttons
		closebutton = gtk.Button(stock=gtk.STOCK_CLOSE)
		closebutton.connect('clicked', self.preferences_close_clicked)
		closebutton.set_flags(gtk.CAN_DEFAULT)

		buttonbox = gtk.HButtonBox()
		buttonbox.set_spacing(10)
		buttonbox.set_layout(gtk.BUTTONBOX_END)
		buttonbox.pack_start(closebutton)

		# Packing everythin into a vbox
		vbox = gtk.VBox()
		vbox.set_border_width(10)
		vbox.set_spacing(10)
		vbox.pack_start(dev_vbox)
		vbox.pack_start(tim_vbox)
		vbox.pack_start(notify_vbox)
		vbox.pack_start(autostart_vbox)
		vbox.pack_start(buttonbox)

		prefs.add(vbox)
		self.dev_entry.grab_focus()
		closebutton.grab_default()

		return prefs
