/* This file is part of lomiri-action-api
 * Copyright 2013 Canonical Ltd.
 *
 * lomiri-action-api is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * lomiri-action-api is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lomiri/action/PreviewRangeParameter>
using namespace lomiri::action;

namespace lomiri {
namespace action {
/*!
 * \class PreviewRangeParameter
 * Provides a range preview parameter.
 * See \ref page_preview-actions for more details.
 */
}
}

// properties

/*!
 * \property float PreviewRangeParameter::maximumValue
 *
 * The maximum value of the range.
 *
 * \initvalue 100.0f
 *
 * \accessors maximumValue(), setMaximumValue()
 *
 * \notify maximumValueChanged()
 */

/*!
 * \property float PreviewRangeParameter::minimumValue
 *
 * The minimum value of the range.
 *
 * \initvalue 0.0f
 *
 * \accessors minimumValue(), setMinimumValue()
 *
 * \notify minimumValueChanged()
 */

/*!
 * \property QString PreviewRangeParameter::text
 *
 * The user visible primary label of the parameter.
 *
 * \initvalue ""
 *
 * \accessors text(), setText()
 *
 * \notify textChanged()
 */

/*!
 * \property float PreviewRangeParameter::value
 *
 * The current value of the range.
 *
 * \initvalue 0.0f
 *
 * \accessors value(), setValue()
 *
 * \notify valueChanged()
 */


//! \private
class Q_DECL_HIDDEN lomiri::action::PreviewRangeParameter::Private {
public:
    QString text;
    float value;
    float min;
    float max;
};

/*!
 * \fn PreviewRangeParameter::PreviewRangeParameter(QObject *parent = 0)
 * \param parent parent QObject or 0
 *
 * Constructs a new PreviewRangeParameter. See the property documentation for
 * default values.
 */
PreviewRangeParameter::PreviewRangeParameter(QObject *parent)
    : PreviewParameter(parent),
      d(new Private())
{
    d->max = 100.0f;
    d->min = 0.0f;
    d->value = 0.0f;
}

PreviewRangeParameter::~PreviewRangeParameter()
{
}

QString
PreviewRangeParameter::text() const
{
    return d->text;
}

void
PreviewRangeParameter::setText(const QString &value)
{
    if (d->text == value)
        return;
    d->text = value;
    emit textChanged(value);
}

float
PreviewRangeParameter::value() const
{
    return d->value;
}

void
PreviewRangeParameter::setValue(float value)
{
    if (qFuzzyCompare(d->value, value))
        return;

    if (!qFuzzyCompare(d->min, value) && value < d->min) {
        qWarning("%s: trying to set range value (%f) below minimum (%f)",
                 __PRETTY_FUNCTION__,
                 value,
                 d->min);
        return;
    }

    if (!qFuzzyCompare(d->max, value) && value > d->max) {
        qWarning("%s: trying to set range value (%f) above maximum (%f)",
                 __PRETTY_FUNCTION__,
                 value,
                 d->max);
        return;
    }

    d->value = value;
    emit valueChanged(value);
}

float
PreviewRangeParameter::minimumValue() const
{
    return d->min;
}

void
PreviewRangeParameter::setMinimumValue(float value)
{
    if (qFuzzyCompare(d->min, value))
        return;

    // make sure min is not bigger than max
    if (!qFuzzyCompare(value, d->max) && value > d->max) {
        qWarning("%s: trying to set minimum value (%f) above maximum (%f)",
                 __PRETTY_FUNCTION__,
                 value,
                 d->max);
        d->min = d->max;
    } else {
        d->min = value;
    }
    emit minimumValueChanged(d->min);

    // make sure that current value is not smaller than min
    // if so, update value also
    if (!qFuzzyCompare(d->value, d->min) && d->value < d->min) {
        setValue(d->min);
    }
}

float
PreviewRangeParameter::maximumValue() const
{
    return d->max;
}

void
PreviewRangeParameter::setMaximumValue(float value)
{
    if (qFuzzyCompare(d->max, value))
        return;

    // make sure max is not smaller than min
    if (!qFuzzyCompare(value, d->min) && value < d->min) {
        qWarning("%s: trying to set maximum value (%f) below minimum (%f)",
                 __PRETTY_FUNCTION__,
                 value,
                 d->min);
        d->max = d->min;
    } else {
        d->max = value;
    }
    emit maximumValueChanged(d->max);

    // make sure that current value is not bigger than max
    // if so, update value also
    if (!qFuzzyCompare(d->value, d->max) && d->value > d->max) {
        setValue(d->max);
    }
}
