%feature("docstring") OT::GeneralizedExponential
R"RAW(Absolute exponential covariance function.

Available constructors:
    GeneralizedExponential(*spatialDim=1*)

    GeneralizedExponential(*scale, p*)

    GeneralizedExponential(*scale, amplitude, p*)


Parameters
----------
spatialDim : int
    Spatial dimension :math:`n`.
    When not fulfilled, the input dimension is equal to the  size of the parameter :math:`\vect{\theta}`.
    By default, equal to 1.
scale : sequence of floats
    Scale coefficient :math:`\vect{\theta}\in \Rset^n`.
    The size of :math:`\vect{\theta}` is the input dimension.
amplitude : sequence of positive floats
    Amplitude of the process :math:`\vect{\sigma}\in \Rset^d`.
    Must be of size equal to 1.
    By default, equal to :math:`[1]`.
p : float, :math:`0<p \leq 2`
    Define the exponent of the euclidean norm that is used within the model.

Notes
-----
The *generalized exponential function* is a stationary covariance function with dimension :math:`d=1`.

We consider the scalar stochastic process :math:`X: \Omega \times\cD \mapsto \Rset`, where :math:`\omega \in \Omega` is an event, :math:`\cD` is a domain of :math:`\Rset^n`.

The  *generalized exponential* function is defined by:

.. math::
    C(\vect{s}, \vect{t}) = \sigma^2 e^{-\left|\left|\dfrac{\vect{s}-\vect{t}}{\vect{\theta}}\right|\right|_{2}^p}, \quad \forall (\vect{s}, \vect{t}) \in \cD

The correlation function :math:`\rho` writes:

.. math::

    \rho(\vect{s}, \vect{t}) = e^{-\left\| \vect{s}-\vect{t} \right\||_{2}^p}, \quad \forall (\vect{s}, \vect{t}) \in \cD


See Also
--------
CovarianceModel

Examples
--------
Create a standard generalized exponential covariance function:

>>> import openturns as ot
>>> covModel = ot.GeneralizedExponential(2)
>>> t = [0.1, 0.3]
>>> s = [0.2, 0.4]
>>> print(covModel(s, t))
[[ 0.868123 ]]
>>> tau = [0.1, 0.3]
>>> print(covModel(tau))
[[ 0.728893 ]]

Create a  generalized exponential covariance function specifying the scale vector and p:

>>> covModel2 = ot.GeneralizedExponential([1.5, 2.5], 1.5)
>>> covModel2bis = ot.GeneralizedExponential([1.5] * 2, 1.5)

Create a  generalized exponential covariance function specifying the scale vector, the amplitude and p:

>>> covModel3 = ot.GeneralizedExponential([1.5, 2.5], [3.5], 1.5)
)RAW"


// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExponential::setP
R"RAW(P accessor.

Parameters
----------
p : int, :math:`p \geq 1`
    Define the norm that is used within the model.)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedExponential::getP
R"RAW(P accessor.

Returns
-------
p : int, :math:`p \geq 1`
    Define the norm that is used within the model.)RAW"

